<?php

namespace Tests\Functional;

use FCAPoland\DealerAPIHelper\DealerLoader;
use FCAPoland\DealerAPIHelper\Dealers;
use PHPUnit\Framework\TestCase;
use Tests\DealerLoaderFactory;
use Tests\Liberator;

/**
 * Class DealersTest
 *
 * These tests require fully functional HTTP server serving API data from any source:
 *  - development
 *  - tests
 *  - production
 *
 * Please check `DEALER_API_URL` constant than can be set in PHPUnit's `bootstrap.php` file.
 *
 * @package Tests\Functional
 */
class DealersTest extends TestCase
{
    /**
     * Change these to match current dealer availability
     */
    private const EXAMPLE_DEALER_SINCOM_1 = '0075291';
    private const EXAMPLE_DEALER_SINCOM_2 = '0075381';
    private const EXAMPLE_DEALER_SINCOM_WITH_SITECODE_1 = '0075291.000';
    private const EXAMPLE_DEALER_SINCOM_WITH_SITECODE_2 = '0075713.000';
    private const EXAMPLE_DEALER_SINCOM_WITH_SITECODE_3 = '0075381.001';
    private const EXAMPLE_OIC_1 = '00000391';
    private const EXAMPLE_RRDI_CODE_1 = 'PL0013J';

    private DealerLoader $dealerLoader;

    protected function setUp(): void
    {
        $this->dealerLoader = DealerLoaderFactory::create();
        if (defined('DEALER_API_URL')) {
            // To make dealerLoader use of possibly defined `DEALER_API_URL` constant:
            $this->dealerLoader->setDealerAPIUrl(DEALER_API_URL);
        }
    }

    public function testWithSINCOMs(): void
    {
        $dealers = new Dealers($this->dealerLoader);
        $sincom = $input = self::EXAMPLE_DEALER_SINCOM_WITH_SITECODE_1;
        $result = $dealers
            ->withSINCOMs([$input])
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertEquals($sincom, $dealer->getCode());
        }
        unset($dealers, $sincom, $input, $result, $dealer);

        $dealers = new Dealers($this->dealerLoader);
        $sincom = $input = self::EXAMPLE_DEALER_SINCOM_WITH_SITECODE_1;
        $result = $dealers
            ->withSINCOMs([$input])
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertEquals($sincom, $dealer->getCode());
        }
        unset($dealers, $sincom, $input, $result, $dealer);

        $dealers = new Dealers($this->dealerLoader);
        $sincoms = $input = [self::EXAMPLE_DEALER_SINCOM_WITH_SITECODE_2, self::EXAMPLE_DEALER_SINCOM_WITH_SITECODE_3];
        $result = $dealers
            ->withSINCOMs($input)
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertContains($dealer->getCode(), $sincoms);
        }
        unset($dealers, $input, $result, $dealer);

        $dealers = new Dealers($this->dealerLoader);
        $sincoms = $input = [self::EXAMPLE_DEALER_SINCOM_2, self::EXAMPLE_DEALER_SINCOM_1];
        $result = $dealers
            ->withSINCOMs($input)
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertContains($dealer->getSINCOM(), $sincoms);
        }
        unset($dealers, $input, $result, $dealer);
    }

    public function testInvalidDealerAPIURL(): void
    {
        $dealerLoader = DealerLoaderFactory::create();
        $dealerLoader->setDealerAPIUrl('http://foo.bar.invalid.url/foo/bar');
        $dealers = new Dealers($dealerLoader);
        $result = $dealers->get();

        $this->assertEmpty($result);
    }

    public function testWithSINCOMsError(): void
    {
        $dealers = new Dealers($this->dealerLoader);
        $this->expectException(\InvalidArgumentException::class);
        $dealers
            ->withSINCOMs(['012345'])
            ->get();
    }

    public function testWithBrands(): void
    {
        $dealers = new Dealers($this->dealerLoader);
        $brands = $input = ['00', '83'];
        $result = $dealers
            ->withBrands($input)
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertContains($dealer->getBrandId(), $brands);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }

    public function testIncludeRZData(): void
    {
        // Revert back to URL, so query parameters will become available
        $dealerLoader = clone $this->dealerLoader;
        $dealerLoader->setDealerAPIUrl('https://api.fcapoland.pl/dealers');
        $dealers = new Dealers($dealerLoader);
        $result = $dealers
            ->withSINCOMs([self::EXAMPLE_DEALER_SINCOM_1])
            ->includeRZData()
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $dealer_liberated = new Liberator($dealer);
            $this->assertArrayHasKey('rzs', $dealer_liberated->properties);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }

    public function testIncludeServices(): void
    {
        // Revert back to URL, so query parameters will become available
        $dealerLoader = clone $this->dealerLoader;
        $dealerLoader->setDealerAPIUrl('https://api.fcapoland.pl/dealers');
        $dealers = new Dealers($dealerLoader);
        $result = $dealers
            ->withSINCOMs([self::EXAMPLE_DEALER_SINCOM_1])
            ->includeServices()
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $dealer_liberated = new Liberator($dealer);
            $this->assertArrayHasKey('services', $dealer_liberated->properties);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }

    public function testIncludeOpeningHours(): void
    {
        // Revert back to URL, so query parameters will become available
        $dealerLoader = clone $this->dealerLoader;
        $dealerLoader->setDealerAPIUrl('https://api.fcapoland.pl/dealers');
        $dealers = new Dealers($dealerLoader);
        $result = $dealers
            ->withSINCOMs([self::EXAMPLE_DEALER_SINCOM_1])
            ->includeOpeningHours()
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $dealer_liberated = new Liberator($dealer);
            $this->assertArrayHasKey('opening_hours', $dealer_liberated->properties);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }

    public function testIncludeActivityCode(): void
    {
        // Revert back to URL, so query parameters will become available
        $dealerLoader = clone $this->dealerLoader;
        $dealerLoader->setDealerAPIUrl('https://api.fcapoland.pl/dealers');
        $dealers = new Dealers($dealerLoader);
        $result = $dealers
            ->withSINCOMs([self::EXAMPLE_DEALER_SINCOM_1])
            ->includeActivityCode()
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $dealer_liberated = new Liberator($dealer);
            $this->assertArrayHasKey('activity_code', $dealer_liberated->properties);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }

    public function testIncludeRelatedEntities(): void
    {
        // Revert back to URL, so query parameters will become available
        $dealerLoader = clone $this->dealerLoader;
        $dealerLoader->setDealerAPIUrl('https://api.fcapoland.pl/dealers');
        $dealers = new Dealers($dealerLoader);
        $result = $dealers
            ->withSINCOMs([self::EXAMPLE_DEALER_SINCOM_1])
            ->includeRelatedEntities()
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $dealer_liberated = new Liberator($dealer);
            $this->assertArrayHasKey('related_entities', $dealer_liberated->properties);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }

    public function testLegalEntityProperty(): void
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withSINCOMs([self::EXAMPLE_DEALER_SINCOM_1]) // To simplify the test: narrow down the tests to single dealer: CENTRUM SPRZEDAŻY FCA POLAND S.A.
            ->includeActivityCode()
            ->get();

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertEquals(self::EXAMPLE_DEALER_SINCOM_1, $dealer->getLegalEntity());

        }
    }

    public function testWithLegalEntities(): void
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withLegalEntities([self::EXAMPLE_DEALER_SINCOM_1])
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertEquals(self::EXAMPLE_DEALER_SINCOM_1, $dealer->getLegalEntity());
        }
    }

    public function testWithMaincodes(): void
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withMainCodes([self::EXAMPLE_DEALER_SINCOM_1])
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertEquals(self::EXAMPLE_DEALER_SINCOM_1, $dealer->getMainCode());
        }
    }

    public function testWithRRDICodes(): void
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withRRDICodes([self::EXAMPLE_RRDI_CODE_1])
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertEquals(self::EXAMPLE_RRDI_CODE_1, $dealer->getRRDI());
        }
    }

    public function testWithOICs(): void
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withOICs([self::EXAMPLE_OIC_1])
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertEquals(self::EXAMPLE_OIC_1, $dealer->getOIC());
        }
    }

    public function testWithoutSubdealers(): void
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withoutSubdealers()
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertFalse($dealer->isSubDealer());
        }
    }

    public function testWithoutSubdealers2(): void
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withoutSubdealers(false)
            ->get();

        $this->assertNotEmpty($result);

        $at_least_one_subdealer = false;

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            if ($dealer->isSubDealer()) {
                $at_least_one_subdealer = true;
                break;
            }
        }

        $this->assertTrue($at_least_one_subdealer);
    }
}
