<?php

namespace FCAPoland\DealerAPIHelper;

class Dealer
{
    public const CODE_INVENDUTO = '0000000';
    public const SINCOM_INVENDUTO = '0000000.000';
    public const RANDOM_DEALER = 'RANDOM';
    public const VIRTUAL_DEALER_FLEET_DEPARTMENT_WARSAW = 'FLEETDEPWAW';

    /**
     * List of object's properties that can be directly returned when invoking `asArray()` method
     */
    protected array $properties = [];

    public function __construct(
        private string $sincom,
        private string $sitecode,
        private string $brand_id,
        private string $type
    ) {
    }

    /**
     * @psalm-api
     */
    public function getSINCOM(): string
    {
        return $this->sincom;
    }

    /**
     * @psalm-api
     */
    public function getSiteCode(): string
    {
        return $this->sitecode;
    }

    /**
     * @psalm-api
     */
    public function getBrandId(): string
    {
        return $this->brand_id;
    }

    /**
     * @psalm-api
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * Return dealer code in form: SINCOM.SITECODE
     *
     * @psalm-api
     */
    public function getCode(): string
    {
        return $this->getSINCOM() . '.' . $this->getSiteCode();
    }

    /**
     * UID (Unique ID) is a composition of:
     *  - SINCOM (7 digits)
     *  - Site code (3 digits)
     *  - brand ID (2 digits)
     *  - service type ("sales" or "aftersales")
     *
     * This function returns this values glued (by default) with dot separator, for example:
     *
     *     0075450.001.00.sales
     *
     *  @psalm-api
     */
    public function getUID(string $separator = '.'): string
    {
        return implode(
            separator: $separator,
            array: [
                $this->getSINCOM(),
                $this->getSiteCode(),
                $this->getBrandId(),
                $this->getType(),
            ]
        );
    }

    /**
     * @psalm-api
     */
    public function getRRDI(): ?string
    {
        return $this->getProperty('rrdi') === null ? null : (string)$this->getProperty('rrdi');
    }

    /**
     * @psalm-api
     */
    public function getName(): ?string
    {
        return $this->getProperty('name') === null ? null : (string)$this->getProperty('name');
    }

    /**
     * @psalm-api
     */
    public function getOtherName(): ?string
    {
        return $this->getProperty('other_name') === null ? null : (string)$this->getProperty('other_name');
    }

    /**
     * @psalm-api
     */
    public function getEmail(): ?string
    {
        return $this->getProperty('email') === null ? null : (string)$this->getProperty('email');
    }

    /**
     * @psalm-api
     */
    public function getWebsite(): ?string
    {
        return $this->getProperty('website') === null ? null : (string)$this->getProperty('website');
    }

    /**
     * @psalm-api
     */
    public function getAddress(): ?Address
    {
        if ($this->getProperty('address') === null) {
            return null;
        }

        return Address::createFromArray((array)$this->getProperty('address'));
    }

    /**
     * OIC looks like: 00000769
     *
     * @psalm-api
     */
    public function getOIC(): ?string
    {
        return $this->getProperty('OIC') === null ? null : (string)$this->getProperty('OIC');
    }

    /**
     * @return array<array-key,string>|null
     * @psalm-api
     */
    public function getPhones(): ?array
    {
        return $this->getProperty('phones') === null ? null : (array)$this->getProperty('phones');
    }

    /**
     * @psalm-api
     */
    public function getFax(): ?string
    {
        return $this->getProperty('fax') === null ? null : (string)$this->getProperty('fax');
    }

    /**
     * @psalm-api
     */
    public function getCoordinates(): ?Coordinates
    {
        $coordinates = $this->getProperty('coordinates');
        if ($coordinates === null) {
            return null;
        }

        return Coordinates::createFromArray((array)$coordinates);
    }

    /**
     * @psalm-api
     */
    public function isSubDealer(): ?bool
    {
        return $this->getProperty('subdealer') === null ? null : (bool)$this->getProperty('subdealer');
    }

    /**
     * @psalm-api
     */
    public function getOpeningHours(): ?OpeningHours
    {
        if ($this->getProperty('opening_hours') === null) {
            return null;
        }

        return OpeningHours::createFromArray((array)$this->getProperty('opening_hours'));
    }

    /**
     * Every entity looks like: "0720011.000.57.aftersales" (like UID used in this library)
     *
     * @return array<array-key,string>|null
     * @psalm-api
     */
    public function getRelatedEntities(): ?array
    {
        return $this->getProperty('related_entities') === null ? null : (array)$this->getProperty('related_entities');
    }

    /**
     * Activity code looks like: "IA"
     *
     * @psalm-api
     */
    public function getActivityCode(): ?string
    {
        return $this->getProperty('activity_code') === null ? null : (string)$this->getProperty('activity_code');
    }

    /**
     * @psalm-api
     */
    public function getServices(): ?Services
    {
        $services = $this->getProperty('services');
        if ($services === null) {
            return null;
        }

        return Services::createFromArray((array)$services);
    }

    /**
     * @psalm-api
     */
    public function getLegalEntity(): ?string
    {
        return $this->getProperty('legal_entity') === null ? null : (string)$this->getProperty('legal_entity');
    }

    /**
     * @psalm-api
     */
    public function getMainCode(): ?string
    {
        return $this->getProperty('maincode') === null ? null : (string)$this->getProperty('maincode');
    }

    /**
     * @psalm-api
     */
    public function getMainDealer(): ?string
    {
        // 'main_dealer' is imported from static text file exported from some not really known system.
        // Take a loot at FCA API's `Task_Dealer_ImportAgencies` class.
        return $this->getProperty('main_dealer') === null ? null : (string)$this->getProperty('main_dealer');
    }

    /**
     * @return ZoneManager[]|null
     * @psalm-api
     */
    public function getZoneManagers(): ?array
    {
        $rzs = $this->getProperty('rzs');
        if (!is_array($rzs)) {
            return null;
        }

        return array_map(
            fn(array $rz, string $type) => ZoneManager::createFromArray(data: array_merge($rz, ['type' => $type])),
            $rzs,
            array_keys($rzs)
        );
    }

    /**
     * @psalm-api
     */
    public function getZoneManager(string $type): ?ZoneManager
    {
        if (!ZoneManager::isTypeValid($type)) {
            throw new \DomainException(
                sprintf(
                    'Invalid Zone Manager type "%s". Valid types are: %s',
                    $type,
                    implode(', ', ZoneManager::getTypes())
                )
            );
        }

        foreach ($this->getZoneManagers() as $zone_manager) {
            if ($zone_manager->getType() === $type) {
                return $zone_manager;
            }
        }

        return null;
    }

    /**
     * @psalm-param array{brand: mixed, code: mixed, sitecode: mixed, type: mixed,...} $properties
     */
    public function setProperties(array $properties): void
    {
        $protected_properties = [
            'code' => null,
            'sitecode' => null,
            'brand' => null,
            'type' => null,
        ];

        $this->properties = array_diff_key($properties, $protected_properties);
    }

    /**
     * Get property like 'code', 'sitecode', 'brand', 'name', 'type', 'address', 'website', 'email', 'phones', etc.
     *
     * @psalm-api
     */
    private function getProperty(string $name): mixed
    {
        return $this->properties[$name] ?? null;
    }

    /**
     * List of RZs that are bound to the dealer. At most one RZ of given type (sales, leasys, stellantis_finance)
     * is allowed.
     * Every RZ has the following properties:
     *
     *  - last_name, the last name of the person, required
     *  - first_name, as above but concerning first name, required
     *  - email, valid e-mail address that in many cases serves as unique key of RZ, required
     *  - phone, phone number, optional, may be empty
     *
     * Returned value is a simple array with following structure:
     *
     * [
     *   [sales] => [
     *       [last_name] => Foo
     *       [first_name] => Bar
     *       [email] => valid@email.address
     *       [phone] => 512345678
     *   ],
     *   [leasys] => [ ... ]
     *   [stellantis_finance] => [ ... ]
     * ]
     *
     * @return array
     * @deprecated Use `getZoneManagers()` or relevant methods for other types of managers (Leasys, Stellantis Finance)
     *             To be removed in next major version of app (4.x)
     * @psalm-api
     *
     */
    public function getRZs(): array
    {
        return (array)$this->getProperty(name: 'rzs');
    }
}
