<?php

namespace FCAPoland\DealerAPIHelper;

use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;
use Psr\SimpleCache\CacheInterface;

class DealerLoader
{
    use LoggerAwareTrait;

    /**
     * Valid URL to API
     */
    private string $dealer_api_url = 'https://api.fcapoland.pl/dealers';

    /**
     * Any query params to be added when fetching dealers data (like 'extra_fields')
     */
    private array $query_params = [];

    /**
     * @var resource
     */
    private $backup_resource;

    private int $backup_ttl;

    private CacheInterface $cache;

    private string $cache_key = 'fca_dealer_api_json';

    private int $cache_expiration = 600; // 10 minutes as the default

    public function __construct(LoggerInterface $logger = null, ?CacheInterface $cache = null)
    {
        $this->logger = $logger ?: new NullLogger();
        $this->cache = $cache ?: new MemoryCache();
    }

    public function fetch()
    {
        $dealers = $this->loadFromCache();
        if (is_null($dealers) && ($dealers = $this->loadFromAPI())) {
            $this->saveToCache(data: $dealers);
        }
        $this->backup(dealers: $dealers);

        return $dealers;
    }

    private function loadFromCache()
    {
        try {
            return $this->cache->get(key: $this->getCacheKey());
        } catch (\Psr\SimpleCache\InvalidArgumentException $e) {
            $this->logger->error(message: $e->getMessage());

            return null;
        }
    }

    private function saveToCache(string $data): void
    {
        try {
            $this->cache->set(key: $this->getCacheKey(), value: $data, ttl: $this->cache_expiration);
        } catch (\Psr\SimpleCache\InvalidArgumentException $e) {
            $this->logger->error(message: $e->getMessage());
        }
    }

    private function loadFromAPI(): string
    {
        try {
            $curl_handle = curl_init();
            $url = $this->dealer_api_url;
            if ($this->query_params) {
                $url .= '?' . http_build_query(data: $this->query_params);
            }
            curl_setopt(handle: $curl_handle, option: CURLOPT_URL, value: $url);
            curl_setopt(handle: $curl_handle, option: CURLOPT_RETURNTRANSFER, value: true);
            curl_setopt(handle: $curl_handle, option: CURLOPT_HEADER, value: false);
            curl_setopt(
                handle: $curl_handle,
                option: CURLOPT_HTTPHEADER,
                value: ['Cache-Control: no-cache', 'Pragma: no-cache']
            );
            $dealers = (string)curl_exec(handle: $curl_handle);
            curl_close(handle: $curl_handle);

            if (!$dealers) {
                $this->logger->error(message: 'Could not load dealers data directly from API');
                $this->logger->info(message: 'Restoring dealers data from backup...');
                $dealers = $this->loadFromBackup();
            }
        } catch (\Exception $e) {
            $this->logger->error(message: $e->getMessage());
            $this->logger->info(message: 'Restoring dealers data from backup...');
            $dealers = $this->loadFromBackup();
        }

        return $dealers;
    }

    /**
     * @psalm-api
     */
    public function setDealerAPIUrl(string $url): void
    {
        $this->dealer_api_url = $url;
    }

    /**
     * @psalm-api
     */
    public function setCache(CacheInterface $cache, ?int $ttl = null): void
    {
        if ($ttl) {
            $this->cache_expiration = $ttl;
        }

        $this->cache = $cache;
    }

    /**
     * @psalm-api
     */
    public function setBackup($resource, $ttl = null): void
    {
        if (!is_resource($resource)) {
            throw new \InvalidArgumentException(message: 'Backup file is not a resource');
        }

        // Purposely or not: rewind the resource
        rewind(stream: $resource);

        $resource_meta_data = stream_get_meta_data(stream: $resource);

        // all other modes support both: reading AND writing ↓
        if (in_array(needle: $resource_meta_data['mode'], haystack: ['r', 'w', 'a', 'x', 'c'])) {
            throw new \InvalidArgumentException(message: 'Backup file must be valid and readable and writeable');
        }
        if (!is_writable(filename: $resource_meta_data['uri']) || !is_readable(filename: $resource_meta_data['uri'])) {
            // check if the underlying data is readable AND writable
            throw new \InvalidArgumentException(message: 'Backup file must be valid and readable and writeable');
        }
        $this->backup_resource = $resource;

        $default = 43200; // 12 hours by default and when errors
        $ttl = (int) $ttl;
        $this->backup_ttl = $ttl === 0 ? $default : $ttl;
    }

    private function loadFromBackup(): string
    {
        if (!$this->backup_resource) {
            $this->logger->warning(message: 'Backup resource undefined. Use `setBackup()`.');

            return '';
        }

        // Skip first line: it is backup creation date in format: YYYY-MM-DD hh:mm:ss
        if (false === fgets(stream: $this->backup_resource)) {
            // Empty file encountered
            $this->logger->warning(message: 'Backup resource is possibly empty.');

            return '';
        }

        // Read rest of the file as the proper backup data
        $backup = stream_get_contents(stream: $this->backup_resource);
        if ($backup === false) {
            $this->logger->warning(message: 'Backup resource contains invalid data.');

            return '';
        }
        if (null === json_decode(json: $backup)) {
            $this->logger->warning(message: 'Backup resource contains invalid data.');

            return '';
        }

        return $backup;
    }

    private function backup(string $dealers): void
    {
        if (!$this->backup_resource) {
            $this->logger->warning(message: 'Backup resource undefined. Use `setBackup()`.');

            return;
        }

        // First line of the backup is the creation date in format: YYYY-MM-DD hh:mm:ss
        $creation_date = strtotime(datetime: fgets($this->backup_resource));

        if (time() - $creation_date < $this->backup_ttl) {
            // Too early to overwrite the backup
            return;
        }

        if (null === json_decode(json: $dealers)) {
            $this->logger->warning(message: 'Received dealers data is invalid and thus will not be saved as backup');

            return;
        }

        rewind(stream: $this->backup_resource);
        ftruncate(stream: $this->backup_resource, size: 0);
        fwrite(stream: $this->backup_resource, data: date('Y-m-d H:i:s') . PHP_EOL);
        fwrite(stream: $this->backup_resource, data: $dealers);
    }

    public function setExtraFields(array $extra_fields): void
    {
        asort($extra_fields); // Sort the fields for the sake of cache key
        $this->query_params['extra_fields'] = implode(separator: ',', array: array_filter($extra_fields));
    }

    public function clearCache(): void
    {
        $this->cache->clear();
    }

    private function getCacheKey(): string
    {
        if ($this->query_params) {
            $query_params = $this->query_params;
            // Sort array to prevent duplicates (different cache keys pointing to the same contents)
            asort($query_params);
            $suffix = md5(serialize($query_params));
        }

        return $this->cache_key . '_' . md5($this->dealer_api_url) . (isset($suffix) ? '_' . $suffix : '');
    }
}
