<?php

namespace FCAPoland\DealerAPIHelper;

class ZoneManager
{
    public const TYPE_SALES = 'sales';
    public const TYPE_LEASYS = 'leasys';
    public const TYPE_STELLANTIS_FINANCE = 'stellantis_finance';

    public function __construct(
        private string $type,
        private string $last_name,
        private string $first_name,
        private string $email,
        private ?string $phone,
    ) {
        if (!self::isTypeValid($type)) {
            throw new \DomainException(
                message: sprintf(
                    '"%s" is not a valid Zone Manager type, valid types are: %s',
                    $type,
                    implode(', ', self::getTypes())
                )
            );
        }
    }

    public function getType(): string
    {
        return $this->type;
    }

    public function getLastName(): string
    {
        return $this->last_name;
    }

    public function getFirstName(): string
    {
        return $this->first_name;
    }

    public function getFullName(): string
    {
        return $this->first_name;
    }

    public function getEmail(): string
    {
        return $this->email;
    }

    public function getPhone(): ?string
    {
        return $this->phone;
    }

    public static function isTypeValid(string $type): bool
    {
        return in_array(needle: $type, haystack: self::getTypes(), strict: true);
    }

    public static function getTypes(): array
    {
        return [
            self::TYPE_SALES,
            self::TYPE_LEASYS,
            self::TYPE_STELLANTIS_FINANCE,
        ];
    }

    public static function createFromArray(array $data): self
    {
        if (!isset($data['type'])) {
            throw new \RuntimeException(message: 'Every Zone Manager needs a "type" attribute');
        }

        return new self(
            type: $data['type'],
            last_name: $data['last_name'] ? (string)$data['last_name'] : null,
            first_name: $data['first_name'] ? (string)$data['first_name'] : null,
            email: $data['email'] ? (string)$data['email'] : null,
            phone: $data['phone'] ? (string)$data['phone'] : null,
        );
    }
}
