<?php

namespace FCAPoland\DealerAPIHelper;

use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;

class Dealers implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    public const EXTRA_FIELDS_RZ = 'rz';
    public const EXTRA_FIELDS_SERVICES = 'services';
    public const EXTRA_FIELDS_OPENING_HOURS = 'opening_hours';
    public const EXTRA_FIELDS_ACTIVITY_CODE = 'activity_code';
    public const EXTRA_FIELDS_RELATED_ENTITIES = 'related_entities';

    private array $brands_map = [
        '00' => 'Fiat',
        '11' => 'Camper',
        '55' => 'Chrysler',
        '56' => 'Dodge',
        '57' => 'Jeep',
        '66' => 'Abarth',
        '70' => 'Lancia',
        '77' => 'Fiat Professional',
        '82' => 'Leapmotor',
        '83' => 'Alfa Romeo',
        '99' => 'Used cars',
    ];

    private array $filters = [
        'brands' => [],
        'sincoms' => [],
        'types' => [],
        'legal_entites' => [],
        'maincodes' => [],
        'rrdi_codes' => [],
        'oics' => [],
        'no_subdealers' => null,
    ];

    private array $extra_fields = [];

    private DealerLoader $dealer_loader;

    /**
     * @psalm-api
     */
    public function __construct(
        DealerLoader $dealer_loader = null,
        ?LoggerInterface $logger = null,
        ?CacheInterface $cache = null
    ) {
        if (!$dealer_loader instanceof DealerLoader) {
            $dealer_loader = new DealerLoader($logger, $cache);
        }
        $this->setLoader(dealer_loader: $dealer_loader);
    }

    /**
     * @psalm-api
     */
    public function setLoader(DealerLoader $dealer_loader): void
    {
        $this->dealer_loader = $dealer_loader;
    }

    /**
     * @psalm-api
     */
    public function withSINCOMs(array $sincoms): static
    {
        // SINCOM parameter MAY be assumed COMPLETE, i.e. consisting of SINCOM and SITECODE, dot-separated,
        // like: "0075467.003" __OR MAY__ include just SINCOM part (7 digits). Others must be treated as INVALID.
        foreach ($sincoms as $sincom) {
            if (!is_scalar($sincom) || !preg_match(pattern: '/^[a-zA-Z0-9]{7}(\.\d{3})?$/', subject: (string)$sincom)) {
                throw new \InvalidArgumentException(
                    message: sprintf(
                        'Invalid dealer SINCOM: %s',
                        is_scalar($sincom) ? (string)$sincom : json_encode($sincom)
                    )
                );
            }
        }
        $this->filters['sincoms'] = $sincoms;

        return $this;
    }

    /**
     * @psalm-api
     */
    public function withBrands(array $brands): static
    {
        $new_brands = [];
        foreach ($brands as $brand) {
            if (!is_scalar($brand)) {
                throw new \InvalidArgumentException(message: 'Invalid brand: ' . json_encode($brand));
            }

            $brand = strtolower((string)$brand);
            if (array_key_exists(key: $brand, array: $this->brands_map)) {
                $new_brands[] = $brand;
                continue;
            }

            foreach ($this->brands_map as $brand_id => $brand_name) {
                if (strtolower((string)$brand_name) === $brand) {
                    $new_brands[] = $brand_id;
                    continue 2;
                }
            }

            throw new \InvalidArgumentException(message: sprintf('Invalid brand: %s', $brand));
        }

        $this->filters['brands'] = $new_brands;

        return $this;
    }

    /**
     * @psalm-api
     */
    public function withTypes(array $types): static
    {
        $valid_types = ['sales', 'aftersales'];
        foreach ($types as $type) {
            if (!in_array(needle: $type, haystack: $valid_types, strict: true)) {
                throw new \InvalidArgumentException(
                    message: sprintf(
                        'Invalid type: %s. Valid types are: %s',
                        json_encode($type),
                        implode(', ', $valid_types)
                    )
                );
            }
        }
        $this->filters['types'] = array_unique($types);

        return $this;
    }

    /**
     * @psalm-api
     */
    public function withLegalEntities(array $sincoms): static
    {
        foreach ($sincoms as $sincom) {
            if (!is_scalar($sincom) || !preg_match(pattern: '/^[a-zA-Z0-9]{7}$/', subject: (string)$sincom)) {
                throw new \InvalidArgumentException(
                    sprintf('Invalid dealer SINCOM: %s', is_scalar($sincom) ? (string)$sincom : json_encode($sincom))
                );
            }
        }
        $this->filters['legal_entites'] = $sincoms;

        return $this;
    }

    /**
     * @psalm-api
     */
    public function withMainCodes(array $maincodes): static
    {
        foreach ($maincodes as $maincode) {
            if (!is_scalar($maincode) || !preg_match(pattern: '/^[a-zA-Z0-9]{7}$/', subject: (string)$maincode)) {
                throw new \InvalidArgumentException(
                    message: sprintf(
                        'Invalid dealer main code (SINCOM): %s',
                        is_scalar($maincode) ? (string)$maincode : json_encode($maincode)
                    )
                );
            }
        }
        $this->filters['maincodes'] = $maincodes;

        return $this;
    }

    /**
     * @psalm-api
     */
    public function withRRDICodes(array $rrdi_codes): static
    {
        foreach ($rrdi_codes as $rrdi_code) {
            if (!is_scalar($rrdi_code)) {
                throw new \InvalidArgumentException(sprintf('Invalid RRDI code: %s', json_encode($rrdi_code)));
            }
        }
        $this->filters['rrdi_codes'] = $rrdi_codes;

        return $this;
    }

    /**
     * @psalm-api
     */
    public function withOICs(array $codes): static
    {
        foreach ($codes as $oic) {
            if (!is_scalar($oic)) {
                throw new \InvalidArgumentException(sprintf('Invalid OIC code type: %s', gettype($oic)));
            }
        }
        $this->filters['oics'] = $codes;

        return $this;
    }

    /**
     * @psalm-api
     */
    public function withoutSubdealers(bool $flag = true): static
    {
        $this->filters['no_subdealers'] = $flag;

        return $this;
    }

    /**
     * @psalm-api
     */
    public function includeRZData($flag = true): static
    {
        $this->extra_fields[self::EXTRA_FIELDS_RZ] = (bool) $flag;

        return $this;
    }

    /**
     * @psalm-api
     */
    public function includeServices($flag = true): static
    {
        $this->extra_fields[self::EXTRA_FIELDS_SERVICES] = (bool) $flag;

        return $this;
    }

    /**
     * @psalm-api
     */
    public function includeOpeningHours($flag = true): static
    {
        $this->extra_fields[self::EXTRA_FIELDS_OPENING_HOURS] = (bool) $flag;

        return $this;
    }

    /**
     * @psalm-api
     */
    public function includeActivityCode($flag = true): static
    {
        $this->extra_fields[self::EXTRA_FIELDS_ACTIVITY_CODE] = (bool) $flag;

        return $this;
    }

    /**
     * @psalm-api
     */
    public function includeRelatedEntities($flag = true): static
    {
        $this->extra_fields[self::EXTRA_FIELDS_RELATED_ENTITIES] = (bool) $flag;

        return $this;
    }

    /**
     * @psalm-api
     * @return Dealer[]
     */
    public function get(): array
    {
        $dealers = [];

        if ($extra_fields = array_filter(array: $this->extra_fields)) {
            $this->dealer_loader->setExtraFields(extra_fields: array_keys($extra_fields));
            unset($extra_fields);
        }

        foreach (
            (array)json_decode(
                json: (string)$this->dealer_loader->fetch(),
                associative: true
            ) as $dealer_properties
        ) {
            $dealer_properties = (array)$dealer_properties;
            if (
                !isset(
                    $dealer_properties['code'],
                    $dealer_properties['sitecode'],
                    $dealer_properties['brand'],
                    $dealer_properties['type']
                )
            ) {
                if ($this->logger instanceof LoggerInterface) {
                    $this->logger->error(
                        message: 'Fetched dealer data incomplete',
                        context: ['data' => $dealer_properties]
                    );
                }
                continue;
            }

            $dealer = new Dealer(
                sincom: $dealer_properties['code'],
                sitecode: $dealer_properties['sitecode'],
                brand_id: $dealer_properties['brand'],
                type: $dealer_properties['type']
            );

            $dealer->setProperties($dealer_properties);
            $dealers[$dealer->getUID()] = $dealer;
        };

        return array_filter(array: $dealers, callback: fn(Dealer $dealer): bool => $this->filter($dealer));
    }

    private function filter(Dealer $dealer): bool
    {
        if ($this->filters['brands'] && !in_array(needle: $dealer->getBrandId(), haystack: $this->filters['brands'])) {
            return false;
        }

        if ($this->filters['sincoms']) {
            $result = false;
            foreach ($this->filters['sincoms'] as $sincom) {
                if (preg_match(pattern: '/^[a-zA-Z0-9]{7}$/', subject: (string)$sincom)) {
                    // Just SINCOM
                    if ($dealer->getSINCOM() === $sincom) {
                        $result = true;
                        break;
                    }
                } elseif ($dealer->getCode() === $sincom) {
                    // SINCOM.SITECODE
                    $result = true;
                    break;
                }
            }

            if (!$result) {
                return false;
            }
            unset($result);
        }

        if (
            $this->filters['types']
            && !in_array(needle: $dealer->getType(), haystack: $this->filters['types'])
        ) {
            return false;
        }

        if (
            $this->filters['legal_entites']
            && !in_array(needle: $dealer->getLegalEntity(), haystack: $this->filters['legal_entites'])
        ) {
            return false;
        }

        if (
            $this->filters['maincodes']
            && !in_array(needle: $dealer->getMainCode(), haystack: $this->filters['maincodes'])
        ) {
            return false;
        }

        if (
            $this->filters['rrdi_codes']
            && !in_array(needle: $dealer->getRRDI(), haystack: $this->filters['rrdi_codes'])
        ) {
            return false;
        }

        if ($this->filters['no_subdealers'] && $dealer->isSubDealer()) {
            return false;
        }

        return !(
            $this->filters['oics']
            && !in_array(needle: $dealer->getOIC(), haystack: $this->filters['oics'])
        );
    }

    /**
     * @psalm-api
     */
    public function clearCache(): void
    {
        $this->dealer_loader->clearCache();
    }
}
