<?php

namespace Tests\Unit;

use FCAPoland\DealerAPIHelper\Dealer;
use FCAPoland\DealerAPIHelper\DealerLoader;
use FCAPoland\DealerAPIHelper\Dealers;
use PHPUnit\Framework\TestCase;
use Tests\Liberator;

class DealersTest extends TestCase
{
    /**
     * Change these to match current dealer availability
     */
    private const EXAMPLE_DEALER_SINCOM_WITH_SITECODE_1 = '0075440.001';
    private const EXAMPLE_DEALER_SINCOM_WITH_SITECODE_2 = '0075366.000';
    private const EXAMPLE_DEALER_SINCOM_1 = '0075373';
    private const EXAMPLE_DEALER_SINCOM_2 = '0075463';

    public function testGet(): void
    {
        $dealers = new Dealers($this->mockLoader());
        $result = $dealers->get();

        $this->assertIsArray($result);

        $dealer_id = key($result);
        $this->assertMatchesRegularExpression('/^[a-zA-Z0-9]{7}\.\d{3}\.\d{2}\.(sales|aftersales)$/', $dealer_id);

        $dealer = array_pop($result);
        $this->assertInstanceOf(Dealer::class, $dealer);
    }

    private function mockLoader()
    {
        // Mock loader to fetch JSON from local file instead of calling external API
        $loader = $this->getMockBuilder(DealerLoader::class)
                       ->onlyMethods(['fetch'])
                       ->getMock();

        // This is fake dealers details that is used as a query result, including 'rzs' field
        $dealers_json = <<<JSON
[
  {
    "code": "0012345",
    "sitecode": "001",
    "brand": "00",
    "name": "FOOBARMOTO SP. Z O.O.",
    "type": "sales",
    "subdealer": false,
    "address": {
      "street": "FOOBAROWSKA 12",
      "post_code": "12-345",
      "city": "Foobarowo",
      "region": {
        "id": 13,
        "code": "LDZ",
        "name": "łódzkie"
      }
    },
    "coordinates": {
      "longitude": "19.494603",
      "latitude": "51.90401"
    },
    "website": "http://www.foo-bar-moto.pl/fiat?locationCode=00000123",
    "email": "f12345s@dealer.fiat.pl",
    "phones": [
      "48123456789",
      "48123456798"
    ],
    "fax": "48123456799",
    "level_of_service": null,
    "rz": {
      "last_name": "SalesRz",
      "first_name": "FirstName",
      "email": "firstname.sales@unknown.com",
      "phone": "512345678"
    },
    "rzs": {
      "sales": {
        "last_name": "SalesRz",
        "first_name": "FirstName",
        "email": "firstname.sales@unknown.com",
        "phone": "512345678"
      },
      "leasys": {
        "last_name": "LeasysRz",
        "first_name": "FirstName",
        "email": "firstname.leasysrz@somedomain.com",
        "phone": ""
      },
      "stellantis_finance": {
        "last_name": "John",
        "first_name": "Doe",
        "email": "john.doe@somedomain.com",
        "phone": ""
      }
    }
  }
]
JSON;
        $loader
            ->expects($this->atLeastOnce())
            ->method('fetch')
            ->willReturn($dealers_json);

        return $loader;
    }

    public function testWithSINCOMs(): void
    {
        $dealers = new Dealers();
        $sincoms = [self::EXAMPLE_DEALER_SINCOM_WITH_SITECODE_1];
        $input = $sincoms;
        $dealers->withSINCOMs($input);
        $dealer_liberated = new Liberator($dealers);
        $this->assertEquals($sincoms, $dealer_liberated->filters['sincoms']);

        $dealers = new Dealers();
        $sincoms = [self::EXAMPLE_DEALER_SINCOM_WITH_SITECODE_1, self::EXAMPLE_DEALER_SINCOM_WITH_SITECODE_2];
        $input = $sincoms;
        $dealers->withSINCOMs($input);
        $dealer_liberated = new Liberator($dealers);
        $this->assertEquals($sincoms, $dealer_liberated->filters['sincoms']);

        $dealers = new Dealers();
        $sincoms = [self::EXAMPLE_DEALER_SINCOM_1, self::EXAMPLE_DEALER_SINCOM_2];
        $input = $sincoms;
        $dealers->withSINCOMs($input);
        $dealer_liberated = new Liberator($dealers);
        $this->assertEquals($sincoms, $dealer_liberated->filters['sincoms']);

        $dealers = new Dealers();
        $sincoms = [self::EXAMPLE_DEALER_SINCOM_1, self::EXAMPLE_DEALER_SINCOM_2];
        $input = $sincoms;
        $dealers
            ->withSINCOMs(['0012345'])
            ->withSINCOMs($input);
        $dealer_liberated = new Liberator($dealers);
        $this->assertEquals($sincoms, $dealer_liberated->filters['sincoms']);
    }

    public function testWithSINCOMs_exception_1(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withSINCOMs(['075463']);
    }

    public function testWithSINCOMs_exception_2(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withSINCOMs(['0754634.00']);
    }

    public function testWithSINCOMs_exception_3(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withSINCOMs(['0754634.001', '020202']);
    }

    public function testWithSINCOMs_exception_4(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withSINCOMs(['']);
    }

    public function testWithSINCOMs_exception_5(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withSINCOMs(['abcdefg']);
    }

    public function testWithBrands(): void
    {
        $dealers = new Dealers();
        $brands = ['00'];
        $input = $brands;
        $dealers->withBrands($input);
        $dealer_liberated = new Liberator($dealers);
        $this->assertEquals($brands, $dealer_liberated->filters['brands']);

        $dealers = new Dealers();
        $brands = ['00', '57'];
        $input = $brands;
        $dealers->withBrands($input);
        $dealer_liberated = new Liberator($dealers);
        $this->assertEquals($brands, $dealer_liberated->filters['brands']);

        $dealers = new Dealers();
        $input = ['00', 'jeep'];
        $dealers->withBrands($input);
        $dealer_liberated = new Liberator($dealers);
        $this->assertEquals($brands, $dealer_liberated->filters['brands']);

        $dealers = new Dealers();
        $dealers
            ->withBrands(['83'])
            ->withBrands($input);
        $dealer_liberated = new Liberator($dealers);
        $this->assertEquals($brands, $dealer_liberated->filters['brands']);
    }

    public function testWithBrands_exception_1(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withBrands(['porsche']);
    }

    public function testWithBrands_exception_2(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withBrands(['fiat', 'porsche']);
    }

    public function testWithBrands_exception_3(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withBrands(['fiat', '990']);
    }

    public function testWithBrands_exception_4(): void
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withBrands(['fiat', '']);
    }

    public function testIncludeRZData(): void
    {
        $dealers = new Dealers();
        $dealers->includeRZData();
        $dealer_liberated = new Liberator($dealers);
        $this->assertTrue($dealer_liberated->extra_fields['rz']);

        $dealers = new Dealers();
        $dealers->includeRZData(false);
        $dealer_liberated = new Liberator($dealers);
        $this->assertFalse($dealer_liberated->extra_fields['rz']);
    }

    public function testIncludeServices(): void
    {
        $dealers = new Dealers();
        $dealers->includeServices();
        $dealer_liberated = new Liberator($dealers);
        $this->assertTrue($dealer_liberated->extra_fields['services']);
        $dealers = new Dealers();
        $dealers->includeServices(false);
        $dealer_liberated = new Liberator($dealers);
        $this->assertFalse($dealer_liberated->extra_fields['services']);
    }

    public function testIncludeOpeningHours(): void
    {
        $dealers = new Dealers();
        $dealers->includeOpeningHours();
        $dealer_liberated = new Liberator($dealers);
        $this->assertTrue($dealer_liberated->extra_fields['opening_hours']);

        $dealers = new Dealers();
        $dealers->includeOpeningHours(false);
        $dealer_liberated = new Liberator($dealers);
        $this->assertFalse($dealer_liberated->extra_fields['opening_hours']);
    }

    public function testIncludeActivityCode(): void
    {
        $dealers = new Dealers();
        $dealers->includeActivityCode();
        $dealer_liberated = new Liberator($dealers);
        $this->assertTrue($dealer_liberated->extra_fields['activity_code']);
        $dealers = new Dealers();
        $dealers->includeActivityCode(false);
        $dealer_liberated = new Liberator($dealers);
        $this->assertFalse($dealer_liberated->extra_fields['activity_code']);
    }

    public function testIncludeRelatedEntities(): void
    {
        $dealers = new Dealers();
        $dealers->includeRelatedEntities();
        $dealer_liberated = new Liberator($dealers);
        $this->assertTrue($dealer_liberated->extra_fields['related_entities']);
        $dealers = new Dealers();
        $dealers->includeRelatedEntities(false);
        $dealer_liberated = new Liberator($dealers);
        $this->assertFalse($dealer_liberated->extra_fields['related_entities']);
    }

    public function tzzzzzzzzzestGetRZs(): void
    {
        $dealers = new Dealers($this->mockLoader());
        $result = $dealers->get();

        /** @var Dealer $dealer */
        $dealer = array_pop($result);

        $rzs = $dealer->getRZs();

        $this->assertIsArray($rzs);
        $this->assertNotEmpty($rzs);
        $this->assertArrayHasKey('sales', $rzs);
        $this->assertArrayHasKey('stellantis_finance', $rzs);
        $this->assertArrayHasKey('leasys', $rzs);

        foreach (['sales', 'stellantis_finance', 'leasys'] as $type) {
            $this->assertIsArray($rzs[$type]);
            $this->assertNotEmpty($rzs[$type]);
            $this->assertArrayHasKey('first_name', $rzs[$type]);
            $this->assertNotEmpty($rzs[$type]['first_name']);
            $this->assertArrayHasKey('last_name', $rzs[$type]);
            $this->assertNotEmpty($rzs[$type]['last_name']);
            $this->assertArrayHasKey('email', $rzs[$type]);
            $this->assertNotEmpty($rzs[$type]['email']);
            $this->assertArrayHasKey('phone', $rzs[$type]);
        }
        unset($type);
    }
}
