<?php

namespace FCAPoland\APIHelper;

use Psr\Cache\CacheItemPoolInterface;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;

class DealerLoader
{
    private $dealer_api_url = 'https://api.fcapoland.pl/dealers';

    private $logger;

    /**
     * @var string
     */
    private $backup_filename;

    /**
     * @var int
     */
    private $backup_ttl;

    /**
     * @var CacheItemPoolInterface
     */
    private $cache;

    /**
     * @var string
     */
    private $cache_key = 'fca_dealer_api_json';

    /**
     * @var \DateTime
     */
    private $cache_expiration;

    public function __construct(LoggerInterface $logger = null)
    {
        $this->logger = $logger ? $logger : new NullLogger();
    }

    public function fetch()
    {
        $dealers = $this->loadFromCache();
        if (is_null($dealers)) {
            if ($dealers = $this->loadFromAPI()) {
                $this->saveToCache($dealers);
            }
        }

        return $dealers;
    }

    private function loadFromCache()
    {
        if ($this->cache instanceof CacheItemPoolInterface) {
            try {
                $dealers = $this->cache->getItem($this->cache_key);
                if ($dealers->isHit()) {
                    return $dealers->get();
                }
            } catch (\Psr\Cache\InvalidArgumentException $e) {
                return null;
            }
        }

        return null;
    }

    private function saveToCache($data)
    {
        if ($this->cache instanceof CacheItemPoolInterface) {
            $this->cache->save(new CacheItem($this->cache_key, $data, $this->cache_expiration));
        }
    }

    private function loadFromAPI()
    {
        try {
            $curl_handle = curl_init();
            curl_setopt($curl_handle, CURLOPT_URL, $this->dealer_api_url);
            curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($curl_handle, CURLOPT_HEADER, false);
            curl_setopt($curl_handle, CURLOPT_HTTPHEADER, ['Cache-Control: no-cache', 'Pragma: no-cache']);
            $dealers = curl_exec($curl_handle);
            curl_close($curl_handle);

            if (!$dealers) {
                $this->logger->error('Could not load dealers data directly from API');
                $this->logger->info('Restoring dealers data from backup...');
                // TODO: Backing up should consider used filters! Just like cache.
                $dealers = $this->loadFromBackup();
            } else {
                // TODO: Backing up should consider used filters! Just like cache.
                $this->backup($dealers);
            }
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
            $this->logger->info('Restoring dealers data from backup...');
            // TODO: Backing up should consider used filters! Just like cache.
            $dealers = $this->loadFromBackup();
        }

        return $dealers;
    }

    /**
     * @param string $url
     */
    public function setDealerAPIUrl($url)
    {
        $this->dealer_api_url = $url;
    }

    public function setCache(CacheItemPoolInterface $cache, $ttl = null)
    {
        $ttl = intval($ttl);
        if ($ttl === 0) {
            // The default: 10 minutes
            $ttl = 10 * 60;
        }

        $this->cache_expiration = new \DateTime('+' . $ttl . ' seconds');
        $this->cache = $cache;
    }

    public function setBackupFile($filename, $ttl = null)
    {
        $default = 43200; // 12 hours by default and when errors
        if (!is_writable($filename)) {
            throw new \InvalidArgumentException('Backup file must be valid and writeable');
        }
        $this->backup_filename = $filename;
        $ttl = intval($ttl);
        $this->backup_ttl = $ttl === 0 ? $default : $ttl;
    }

    private function loadFromBackup()
    {
        // TODO: Backing up should consider used filters! Just like cache.
        if (!$this->backup_filename) {
            $this->logger->warning('Backup file undefined. Use `setBackupFile()`.');
            return '';
        }

        $backup = file_get_contents($this->backup_filename);
        if (null === json_decode($backup)) {
            $this->logger->warning('Backup file contains invalid data.');
            return '';
        }

        return $backup;
    }

    private function backup($dealers)
    {
        // TODO: Backing up should consider used filters! Just like cache.
        if (!$this->backup_filename) {
            $this->logger->warning('Backup file undefined. Use `setBackupFile()`.');
            return;
        }

        if (file_exists($this->backup_filename)
            and filesize($this->backup_filename) > 0
            and (time() - filemtime($this->backup_filename) < $this->backup_ttl)
        ) {
            // Too early to overwrite the backup
            return;
        }

        if (null === json_decode($dealers)) {
            $this->logger->warning('Received dealers data is invalid and thus will not be saved as backup');
            return;
        }

        file_put_contents($this->backup_filename, (string)$dealers);
        touch($this->backup_filename);
    }
}
