<?php

namespace FCAPoland\APIHelper\MemoryCache;

use FCAPoland\APIHelper\CacheException;
use FCAPoland\APIHelper\CacheItem;
use Psr\Cache\CacheItemInterface;
use Psr\Cache\CacheItemPoolInterface;

/**
 * Class ItemPool
 *
 * Simple cache pool mainly for testing purposes.
 *
 * @package FCAPoland\APIHelper\MemoryCache
 */
final class ItemPool implements CacheItemPoolInterface
{
    /**
     * @var array
     */
    private $storage;

    public function __construct(array &$storage = [])
    {
        $this->storage = &$storage;
    }

    /**
     * @inheritdoc
     */
    public function getItem($key)
    {
        if (!is_string($key)) {
            throw new CacheException('Cache key must be a string!');
        }

        $cache_item = new CacheItem($key, null);
        $cache_item->setIsHit(false);

        if (array_key_exists($key, $this->storage)) {
            $cache_item->set($this->storage[$key]);
            $cache_item->setIsHit(true);
        }

        return $cache_item;
    }

    /**
     * @inheritdoc
     */
    public function getItems(array $keys = array())
    {
        foreach ($keys as $key) {
            yield $this->getItem($key);
        }
    }

    /**
     * @inheritdoc
     */
    public function hasItem($key)
    {
        if (!is_string($key)) {
            throw new CacheException('Cache key must be a string!');
        }

        return array_key_exists($key, $this->storage);
    }

    /**
     * @inheritdoc
     */
    public function clear()
    {
        $this->storage = [];

        return true;
    }

    /**
     * @inheritdoc
     */
    public function deleteItem($key)
    {
        if (!is_string($key)) {
            throw new CacheException('Cache key must be a string!');
        }

        unset($this->storage[$key]);

        return true;
    }

    /**
     * @inheritdoc
     */
    public function deleteItems(array $keys)
    {
        foreach ($keys as $key) {
            $this->deleteItem($key);
        }

        return true;
    }

    /**
     * @inheritdoc
     */
    public function save(CacheItemInterface $item)
    {
        $this->storage[$item->getKey()] = $item;

        return true;
    }

    /**
     * @inheritdoc
     */
    public function saveDeferred(CacheItemInterface $item)
    {
        return $this->save($item);
    }

    /**
     * @inheritdoc
     */
    public function commit()
    {
        return true;
    }
}
