<?php

namespace FCAPoland\APIHelper;

class Dealer
{
    const CODE_INVENDUTO = '0000000';
    const SINCOM_INVENDUTO = '0000000.000';
    const RANDOM_DEALER = 'RANDOM';
    const VIRTUAL_DEALER_FLEET_DEPARTMENT_WARSAW = 'FLEETDEPWAW';

    /**
     * List of object's properties that can be directly returned when invoking `asArray()` method
     *
     * @var array
     */
    protected $properties = [];

    private $sincom;

    private $sitecode;

    private $brand_id;

    private $type;

    /**
     * Presence state. False for dealers that could not be found, true otherwise.
     *
     * @var bool
     */
    private $loaded = false;

    public function __construct($sincom, $sitecode, $brand_id, $type)
    {
        $this->sincom = $sincom;
        $this->sitecode = $sitecode;
        $this->brand_id = $brand_id;
        $this->type = $type;
    }

    /**
     * Return dealer code in form: SINCOM.SITECODE
     *
     * @return string
     */
    public function getCode()
    {
        if ($sincom = $this->getSINCOM() and $sitecode = $this->getSiteCode()) {
            return $sincom . '.' . $sitecode;
        }

        return null;
    }

    public function getSINCOM()
    {
        return $this->sincom;
    }

    public function getSiteCode()
    {
        return $this->sitecode;
    }

    /**
     * UID (Unique ID) is a composition of:
     *  - SINCOM (7 digits)
     *  - Site code (3 digits)
     *  - brand ID (2 digits)
     *  - service type ("sales" or "aftersales"
     *
     * This function returns this values glued (by default) with dot separator, for example:
     *
     *     0075450.001.00.sales
     *
     * @param string $separator
     *
     * @return string
     */
    public function getUID($separator = '.')
    {
        if ($sincom = $this->getSINCOM()
            and $sitecode = $this->getSiteCode()
            and $brand_id = $this->getBrandId()
            and $type = $this->getType()
        ) {
            return join($separator, [$sincom, $sitecode, $brand_id, $type]);
        }

        return null;
    }

    public function getBrandId()
    {
        return $this->brand_id;
    }

    public function getType()
    {
        return $this->type;
    }

    public function getName()
    {
        return $this->getProperty('name');
    }

    /**
     * Get property like 'code', 'sitecode', 'brand', 'name', 'type', 'address', 'website', 'email', 'phones', etc.
     *
     * @param $name
     *
     * @return mixed|null
     */
    private function getProperty($name)
    {
        return array_key_exists($name, $this->properties) ? $this->properties[$name] : null;
    }

    public function getEmail()
    {
        return $this->getProperty('email');
    }

    public function getWebsite()
    {
        return $this->getProperty('website');
    }

    public function getAddressStreet()
    {
        $address = (array)$this->getAddress();

        return array_key_exists('street', $address) ? $address['street'] : false;
    }

    public function getAddress()
    {
        return $this->getProperty('address');
    }

    public function getAddressCity()
    {
        $address = (array)$this->getAddress();

        return array_key_exists('city', $address) ? $address['city'] : false;
    }

    public function getAddressPostCode()
    {
        $address = (array)$this->getAddress();

        return array_key_exists('post_code', $address) ? $address['post_code'] : false;
    }

    public function getAddressRegionName()
    {
        $region = (array)$this->getAddressRegion();

        return array_key_exists('name', $region) ? $region['name'] : false;
    }

    public function getAddressRegion()
    {
        $address = (array)$this->getAddress();

        return array_key_exists('region', $address) ? $address['region'] : false;
    }

    public function getPhones()
    {
        return (array)$this->getProperty('phones');
    }

    public function getFax()
    {
        return $this->getProperty('fax');
    }

    public function getCoordinates()
    {
        return (array)$this->getProperty('coordinates');
    }

    public function getCoordinateLongitude()
    {
        $coordinates = $this->getCoordinates();

        return array_key_exists('longitude', $coordinates) ? $coordinates['longitude'] : false;
    }

    public function getCoordinateLatitude()
    {
        $coordinates = $this->getCoordinates();

        return array_key_exists('latitude', $coordinates) ? $coordinates['latitude'] : false;
    }

    public function hasSalesService()
    {
        $services = (array)$this->getProperty('services');

        return array_key_exists('sales', $services) and $services['sales'];
    }

    public function isSubDealer()
    {
        return $this->getProperty('subdealer');
    }

    public function getRZFirstName()
    {
        $rz_details = (array)$this->getRZ();

        return array_key_exists('first_name', $rz_details) ? $rz_details['first_name'] : null;
    }

    public function getRZ()
    {
        return $this->getProperty('rz');
    }

    public function getRZLastName()
    {
        $rz_details = (array)$this->getRZ();

        return array_key_exists('last_name', $rz_details) ? $rz_details['last_name'] : null;
    }

    public function getRZEmail()
    {
        $rz_details = (array)$this->getRZ();

        return array_key_exists('email', $rz_details) ? $rz_details['email'] : null;
    }

    public function getRZPhone()
    {
        $rz_details = (array)$this->getRZ();

        return array_key_exists('phone', $rz_details) ? $rz_details['phone'] : null;
    }

    public function getOpeningHours()
    {
        return (array)$this->getProperty('opening_hours');
    }

    public function getOpeningHoursForDate($date)
    {
        if (!is_string($date) and !is_int($date)) {
            throw new \InvalidArgumentException('Date must be given either as timestamp (integer) or date-friendly ' .
                                                'string (YYYY-MM-DD), "' . gettype($date) . '" given');
        }

        if (is_string($date)) {
            $parsed_date = strtotime($date);
            if ($parsed_date === false) {
                throw new \InvalidArgumentException('Date "' . $date . '" is invalid. Use YYYY-MM-DD format.');
            }
            $date = $parsed_date;
            unset($parsed_date);
        }

        $date = date('Y-m-d', $date);
        $opening_hours = $this->getOpeningHours();

        foreach ($opening_hours as $day) {
            if (array_key_exists('date', $day) and $day['date'] == $date) {
                return $day;
            }
        }

        return false;
    }

    public function isLoaded()
    {
        return $this->loaded;
    }

    public function setLoaded($flag)
    {
        $this->loaded = (bool)$flag;
    }

    public function setProperties($properties)
    {
        $protected_properties = [
            'code'     => null,
            'sitecode' => null,
            'brand'    => null,
            'type'     => null,
        ];

        $this->properties = array_diff_key($properties, $protected_properties);
    }
}
