<?php

namespace FCAPoland\APIHelper;

use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;

class Dealers implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    private $brands_map = [
        '00' => 'Fiat',
        '11' => 'Camper',
        '55' => 'Chrysler',
        '56' => 'Dodge',
        '57' => 'Jeep',
        '66' => 'Abarth',
        '70' => 'Lancia',
        '77' => 'Fiat Professional',
        '83' => 'Alfa Romeo',
        '99' => 'Used cars',
    ];

    private $filters = [
        'brands' => [],
        'sincoms' => [],
        'types' => [],
    ];

    /**
     * @var DealerLoader
     */
    private $dealer_loader;

    public function __construct(DealerLoader $dealer_loader = null)
    {
        if (!($dealer_loader instanceof DealerLoader)) {
            $dealer_loader = new DealerLoader();
        }
        $this->setLoader($dealer_loader);
    }

    public function setLoader(DealerLoader $dealer_loader)
    {
        $this->dealer_loader = $dealer_loader;
    }

    public function withSINCOMs(array $sincoms)
    {
        // SINCOM parameter MAY be assumed COMPLETE, i.e. consisting of SINCOM and SITECODE, dot-separated,
        // like: "0075467.003" __OR MAY__ include just SINCOM part (7 digits). Others must be treated as INVALID.
        foreach ($sincoms as $sincom) {
            if (!is_scalar($sincom) or !preg_match('/^\d{7}(\.\d{3})?$/', (string)$sincom)) {
                throw new \InvalidArgumentException(
                    'Invalid dealer SINCOM: ' . is_scalar($sincom) ? (string)$sincom : json_encode($sincom)
                );
            }
        }
        $this->filters['sincoms'] = $sincoms;

        return $this;
    }

    public function withBrands(array $brands)
    {
        $new_brands = [];
        foreach ($brands as $brand) {
            if (!is_scalar($brand)) {
                throw new \InvalidArgumentException('Invalid brand: ' . json_encode($brand));
            }

            $brand = strtolower((string)$brand);
            if (array_key_exists($brand, $this->brands_map)) {
                $new_brands[] = $brand;
                continue;
            }

            foreach ($this->brands_map as $brand_id => $brand_name) {
                if (strtolower($brand_name) === $brand) {
                    $new_brands[] = $brand_id;
                    continue 2;
                }
            }

            throw new \InvalidArgumentException('Invalid brand: ' . (string)$brand);
        }

        $this->filters['brands'] = $new_brands;

        return $this;
    }

    public function withTypes(array $types)
    {
        $new_types = [];
        $valid_types = ['sales', 'aftersales'];
        foreach ($types as $type) {
            if (!in_array($type, $valid_types, true)) {
                throw new \InvalidArgumentException(
                    'Invalid type: ' . json_encode($type) . '. Valid types are: ' . join(', ', $valid_types)
                );
            }
        }
        $this->filters['types'] = $new_types;
        return $this;
    }

    /**
     * @return Dealer[]
     */
    public function get()
    {
        $dealers = [];

        foreach ((array)json_decode($this->dealer_loader->fetch(), true) as $dealer_properties) {
            $dealer_properties = (array)$dealer_properties;
            if (!isset(
                $dealer_properties['code'],
                $dealer_properties['sitecode'],
                $dealer_properties['brand'],
                $dealer_properties['type']
            )) {
                if ($this->logger instanceof LoggerInterface) {
                    $this->logger->error('Fetched dealer data incomplete', ['data' => $dealer_properties]);
                }
                continue;
            }

            $dealer = new Dealer(
                $dealer_properties['code'],
                $dealer_properties['sitecode'],
                $dealer_properties['brand'],
                $dealer_properties['type']
            );

            $dealer->setProperties($dealer_properties);
            $dealer->setLoaded(true);
            $dealers[$dealer->getUID()] = $dealer;
        };

        return array_filter($dealers, function (Dealer $dealer) {
            return $this->filter($dealer);
        });
    }

    private function filter(Dealer $dealer)
    {
        if ($this->filters['brands']
            and !in_array($dealer->getBrandId(), $this->filters['brands'])
        ) {
            return false;
        }

        if ($this->filters['sincoms']) {
            foreach ($this->filters['sincoms'] as $sincom) {
                if (preg_match('/^\d{7}$/', $sincom)) {
                    // Just SINCOM
                    return $dealer->getSINCOM() === $sincom;
                } else {
                    // SINCOM.SITECODE
                    return $dealer->getCode() === $sincom;
                }
            }

            return false;
        }

        if ($this->filters['types']
            and !in_array($dealer->getType(), $this->filters['types'])
        ) {
            return false;
        }

        return true;
    }
}
