<?php

namespace Tests\Functional;

use FCAPoland\APIHelper\DealerLoader;
use FCAPoland\APIHelper\Dealers;

/**
 * Class DealersTest
 *
 * These tests require fully functional HTTP server serving API data from any source:
 *  - development
 *  - tests
 *  - production
 *
 * Please check `DEALER_API_URL` constant than can be set in PHPUnit's `bootstrap.php` file.
 *
 * @package Tests\Functional
 */
class DealersTest extends \PHPUnit_Framework_TestCase
{
    /**
     * @var DealerLoader
     */
    private $dealerLoader;

    protected function setUp()
    {
        parent::setUp();

        $this->dealerLoader = new DealerLoader();
        if (defined('DEALER_API_URL')) {
            // To make dealerLoader use of possibly defined `DEALER_API_URL` constant:
            $this->dealerLoader->setDealerAPIUrl(DEALER_API_URL);
        }
    }

    public function testWithSINCOMs()
    {
        $dealers = new Dealers($this->dealerLoader);
        $sincom = $input = '0075611.000';
        $result = $dealers
            ->withSINCOMs([$input])
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\APIHelper\Dealer $dealer */
            $this->assertEquals($sincom, $dealer->getCode());
        }
        unset($dealers, $sincom, $input, $result, $dealer);

        $dealers = new Dealers($this->dealerLoader);
        $sincom = $input = '0075220.001';
        $result = $dealers
            ->withSINCOMs([$input])
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\APIHelper\Dealer $dealer */
            $this->assertEquals($sincom, $dealer->getCode());
        }
        unset($dealers, $sincom, $input, $result, $dealer);

        $dealers = new Dealers($this->dealerLoader);
        $sincoms = $input = ['0720033.001', '0075366.000'];
        $result = $dealers
            ->withSINCOMs($input)
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\APIHelper\Dealer $dealer */
            $this->assertContains($dealer->getCode(), $sincoms);
        }
        unset($dealers, $sincom, $input, $result, $dealer);

        $dealers = new Dealers($this->dealerLoader);
        $sincoms = $input = ['0075373', '0075463'];
        $result = $dealers
            ->withSINCOMs($input)
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\APIHelper\Dealer $dealer */
            $this->assertContains($dealer->getSINCOM(), $sincoms);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }

    public function testInvalidDealerAPIURL()
    {
        $dealerLoader = new DealerLoader();
        $dealerLoader->setDealerAPIUrl('http://foo.bar.invalid.url/foo/bar');
        $dealers = new Dealers($dealerLoader);
        $result = $dealers->get();

        $this->assertEmpty($result);
    }

    public function testWithSINCOMsError()
    {
        $dealers = new Dealers($this->dealerLoader);
        $this->expectException(\InvalidArgumentException::class);
        $dealers
            ->withSINCOMs(['012345'])
            ->get();
    }

    public function testWithBrands()
    {
        $dealers = new Dealers($this->dealerLoader);
        $brands = $input = ['00', '83'];
        $result = $dealers
            ->withBrands($input)
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\APIHelper\Dealer $dealer */
            $this->assertContains($dealer->getBrandId(), $brands);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }
}
