<?php

namespace FCAPoland\DealerAPIHelper;

use Psr\SimpleCache\CacheInterface;

/**
 * Class MemoryCache
 *
 * Simple in-memory cache pool mainly for testing purposes.
 * It is flushed at the end of the script.
 *
 * @package FCAPoland\DealerAPIHelper
 */
final class MemoryCache implements CacheInterface
{
    /**
     * @var array
     */
    private $storage;

    public function __construct(array &$storage = [])
    {
        $this->storage = &$storage;
    }

    /**
     * @inheritdoc
     */
    public function get($key, $default = null)
    {
        if (!is_string($key)) {
            throw new CacheException('Cache key must be a string!');
        }

        if (array_key_exists($key, $this->storage)) {
            return $this->storage[$key];
        }

        return $default;
    }

    /**
     * @inheritdoc
     */
    public function getMultiple($keys, $default = null)
    {
        foreach ($keys as $key) {
            yield $this->get($key);
        }
    }

    /**
     * @inheritdoc
     */
    public function has($key)
    {
        if (!is_string($key)) {
            throw new CacheException('Cache key must be a string!');
        }

        return array_key_exists($key, $this->storage);
    }

    /**
     * @inheritdoc
     */
    public function clear()
    {
        $this->storage = [];

        return true;
    }

    /**
     * @inheritdoc
     */
    public function delete($key)
    {
        if (!is_string($key)) {
            throw new CacheException('Cache key must be a string!');
        }

        unset($this->storage[$key]);

        return true;
    }

    /**
     * @inheritdoc
     */
    public function deleteMultiple($keys)
    {
        foreach ($keys as $key) {
            $this->delete($key);
        }

        return true;
    }

    /**
     * @inheritdoc
     */
    public function set($key, $value, $ttl = null)
    {
        $this->storage[$key] = $value;
    }

    /**
     * @inheritdoc
     */
    public function setMultiple($values, $ttl = null)
    {
        foreach ($values as $key => $value) {
            $this->set($key, $value, $ttl);
        }

        return true;
    }
}
