<?php

namespace Tests\Functional;

use FCAPoland\DealerAPIHelper\DealerLoader;
use FCAPoland\DealerAPIHelper\Dealers;

/**
 * Class DealersTest
 *
 * These tests require fully functional HTTP server serving API data from any source:
 *  - development
 *  - tests
 *  - production
 *
 * Please check `DEALER_API_URL` constant than can be set in PHPUnit's `bootstrap.php` file.
 *
 * @package Tests\Functional
 */
class DealersTest extends \PHPUnit_Framework_TestCase
{
    /**
     * @var DealerLoader
     */
    private $dealerLoader;

    protected function setUp()
    {
        parent::setUp();

        $this->dealerLoader = new DealerLoader();
        if (defined('DEALER_API_URL')) {
            // To make dealerLoader use of possibly defined `DEALER_API_URL` constant:
            $this->dealerLoader->setDealerAPIUrl(DEALER_API_URL);
        }
    }

    public function testWithSINCOMs()
    {
        $dealers = new Dealers($this->dealerLoader);
        $sincom = $input = '0075206.000';
        $result = $dealers
            ->withSINCOMs([$input])
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertEquals($sincom, $dealer->getCode());
        }
        unset($dealers, $sincom, $input, $result, $dealer);

        $dealers = new Dealers($this->dealerLoader);
        $sincom = $input = '0075220.001';
        $result = $dealers
            ->withSINCOMs([$input])
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertEquals($sincom, $dealer->getCode());
        }
        unset($dealers, $sincom, $input, $result, $dealer);

        $dealers = new Dealers($this->dealerLoader);
        $sincoms = $input = ['0720033.001', '0075366.000'];
        $result = $dealers
            ->withSINCOMs($input)
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertContains($dealer->getCode(), $sincoms);
        }
        unset($dealers, $sincom, $input, $result, $dealer);

        $dealers = new Dealers($this->dealerLoader);
        $sincoms = $input = ['0075373', '0075463'];
        $result = $dealers
            ->withSINCOMs($input)
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertContains($dealer->getSINCOM(), $sincoms);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }

    public function testInvalidDealerAPIURL()
    {
        $dealerLoader = new DealerLoader();
        $dealerLoader->setDealerAPIUrl('http://foo.bar.invalid.url/foo/bar');
        $dealers = new Dealers($dealerLoader);
        $result = $dealers->get();

        $this->assertEmpty($result);
    }

    public function testWithSINCOMsError()
    {
        $dealers = new Dealers($this->dealerLoader);
        $this->expectException(\InvalidArgumentException::class);
        $dealers
            ->withSINCOMs(['012345'])
            ->get();
    }

    public function testWithBrands()
    {
        $dealers = new Dealers($this->dealerLoader);
        $brands = $input = ['00', '83'];
        $result = $dealers
            ->withBrands($input)
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertContains($dealer->getBrandId(), $brands);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }

    public function testIncludeRZData()
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withSINCOMs(['0075206']) // To simplify the test: narrow down the tests to single dealer: CENTRUM SPRZEDAŻY FCA POLAND S.A.
            ->includeRZData()
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $refl = new \ReflectionObject($dealer);
            $properties_prop = $refl->getProperty('properties');
            $properties_prop->setAccessible(true);
            $v = $properties_prop->getValue($dealer);
            $this->assertArrayHasKey('rz', $v);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }

    public function testIncludeServices()
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withSINCOMs(['0075206']) // To simplify the test: narrow down the tests to single dealer: CENTRUM SPRZEDAŻY FCA POLAND S.A.
            ->includeServices()
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $refl = new \ReflectionObject($dealer);
            $properties_prop = $refl->getProperty('properties');
            $properties_prop->setAccessible(true);
            $v = $properties_prop->getValue($dealer);
            $this->assertArrayHasKey('services', $v);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }

    public function testIncludeOpeningHours()
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withSINCOMs(['0075206']) // To simplify the test: narrow down the tests to single dealer: CENTRUM SPRZEDAŻY FCA POLAND S.A.
            ->includeOpeningHours()
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $refl = new \ReflectionObject($dealer);
            $properties_prop = $refl->getProperty('properties');
            $properties_prop->setAccessible(true);
            $v = $properties_prop->getValue($dealer);
            $this->assertArrayHasKey('opening_hours', $v);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }

    public function testIncludeActivityCode()
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withSINCOMs(['0075206']) // To simplify the test: narrow down the tests to single dealer: CENTRUM SPRZEDAŻY FCA POLAND S.A.
            ->includeActivityCode()
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $refl = new \ReflectionObject($dealer);
            $properties_prop = $refl->getProperty('properties');
            $properties_prop->setAccessible(true);
            $v = $properties_prop->getValue($dealer);
            $this->assertArrayHasKey('activity_code', $v);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }

    public function testIncludeRelatedEntities()
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withSINCOMs(['0075206']) // To simplify the test: narrow down the tests to single dealer: CENTRUM SPRZEDAŻY FCA POLAND S.A.
            ->includeActivityCode()
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $refl = new \ReflectionObject($dealer);
            $properties_prop = $refl->getProperty('properties');
            $properties_prop->setAccessible(true);
            $v = $properties_prop->getValue($dealer);
            $this->assertArrayHasKey('related_entities', $v);
        }
        unset($dealers, $sincom, $input, $result, $dealer);
    }

    public function testLegalEntityProperty()
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withSINCOMs(['0075206']) // To simplify the test: narrow down the tests to single dealer: CENTRUM SPRZEDAŻY FCA POLAND S.A.
            ->includeActivityCode()
            ->get();

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertEquals('0075206', $dealer->getLegalEntity());

        }
    }

    public function testWithLegalEntities()
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withLegalEntities(['0075206'])
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertEquals('0075206', $dealer->getLegalEntity());
        }
    }

    public function testWithoutSubdealers()
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withoutSubdealers()
            ->get();

        $this->assertNotEmpty($result);

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            $this->assertFalse($dealer->isSubDealer());
        }
    }

    public function testWithoutSubdealers2()
    {
        $dealers = new Dealers($this->dealerLoader);
        $result = $dealers
            ->withoutSubdealers(false)
            ->get();

        $this->assertNotEmpty($result);

        $at_least_one_subdealer = false;

        foreach ($result as $dealer) {
            /** @var \FCAPoland\DealerAPIHelper\Dealer $dealer */
            if ($dealer->isSubDealer()) {
                $at_least_one_subdealer = true;
                break;
            }
        }

        $this->assertTrue($at_least_one_subdealer);
    }
}
