<?php

namespace FCAPoland\DealerAPIHelper;

use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;

class Dealers implements LoggerAwareInterface
{
    use LoggerAwareTrait;

    const EXTRA_FIELDS_RZ = 'rz';
    const EXTRA_FIELDS_SERVICES = 'services';
    const EXTRA_FIELDS_OPENING_HOURS = 'opening_hours';
    const EXTRA_FIELDS_ACTIVITY_CODE = 'activity_code';
    const EXTRA_FIELDS_RELATED_ENTITIES = 'related_entities';

    private $brands_map = [
        '00' => 'Fiat',
        '11' => 'Camper',
        '55' => 'Chrysler',
        '56' => 'Dodge',
        '57' => 'Jeep',
        '66' => 'Abarth',
        '70' => 'Lancia',
        '77' => 'Fiat Professional',
        '83' => 'Alfa Romeo',
        '99' => 'Used cars',
    ];

    private $filters = [
        'brands' => [],
        'sincoms' => [],
        'types' => [],
        'legal_entites' => [],
        'maincodes' => [],
        'oics' => [],
        'no_subdealers' => null,
    ];

    private $extra_fields = [];

    /**
     * @var DealerLoader
     */
    private $dealer_loader;

    public function __construct(DealerLoader $dealer_loader = null)
    {
        if (!($dealer_loader instanceof DealerLoader)) {
            $dealer_loader = new DealerLoader();
        }
        $this->setLoader($dealer_loader);
    }

    public function setLoader(DealerLoader $dealer_loader)
    {
        $this->dealer_loader = $dealer_loader;
    }

    public function withSINCOMs(array $sincoms)
    {
        // SINCOM parameter MAY be assumed COMPLETE, i.e. consisting of SINCOM and SITECODE, dot-separated,
        // like: "0075467.003" __OR MAY__ include just SINCOM part (7 digits). Others must be treated as INVALID.
        foreach ($sincoms as $sincom) {
            if (!is_scalar($sincom) or !preg_match('/^\d{7}(\.\d{3})?$/', (string)$sincom)) {
                throw new \InvalidArgumentException(
                    'Invalid dealer SINCOM: ' . is_scalar($sincom) ? (string)$sincom : json_encode($sincom)
                );
            }
        }
        $this->filters['sincoms'] = $sincoms;

        return $this;
    }

    public function withBrands(array $brands)
    {
        $new_brands = [];
        foreach ($brands as $brand) {
            if (!is_scalar($brand)) {
                throw new \InvalidArgumentException('Invalid brand: ' . json_encode($brand));
            }

            $brand = strtolower((string)$brand);
            if (array_key_exists($brand, $this->brands_map)) {
                $new_brands[] = $brand;
                continue;
            }

            foreach ($this->brands_map as $brand_id => $brand_name) {
                if (strtolower($brand_name) === $brand) {
                    $new_brands[] = $brand_id;
                    continue 2;
                }
            }

            throw new \InvalidArgumentException('Invalid brand: ' . (string)$brand);
        }

        $this->filters['brands'] = $new_brands;

        return $this;
    }

    public function withTypes(array $types)
    {
        $valid_types = ['sales', 'aftersales'];
        foreach ($types as $type) {
            if (!in_array($type, $valid_types, true)) {
                throw new \InvalidArgumentException(
                    'Invalid type: ' . json_encode($type) . '. Valid types are: ' . join(', ', $valid_types)
                );
            }
        }
        $this->filters['types'] = array_unique($types);

        return $this;
    }

    public function withLegalEntities(array $sincoms)
    {
        foreach ($sincoms as $sincom) {
            if (!is_scalar($sincom) or !preg_match('/^\d{7}$/', (string)$sincom)) {
                throw new \InvalidArgumentException(
                    'Invalid dealer SINCOM: ' . is_scalar($sincom) ? (string)$sincom : json_encode($sincom)
                );
            }
        }
        $this->filters['legal_entites'] = $sincoms;

        return $this;
    }

    public function withMainCodes(array $maincodes)
    {
        foreach ($maincodes as $maincode) {
            if (!is_scalar($maincode) or !preg_match('/^\d{7}$/', (string)$maincode)) {
                throw new \InvalidArgumentException(
                    'Invalid dealer main code (SINCOM): ' . is_scalar($maincode) ? (string)$maincode : json_encode($maincode)
                );
            }
        }
        $this->filters['maincodes'] = $maincodes;

        return $this;
    }

    public function withOICs(array $codes)
    {
        foreach ($codes as $oic) {
            if (!is_scalar($oic)) {
                throw new \InvalidArgumentException('Invalid OIC code type: ' . gettype($oic));
            }
        }
        $this->filters['oics'] = $codes;

        return $this;
    }

    public function withoutSubdealers($flag = true)
    {
        if (!is_bool($flag)) {
            throw new \InvalidArgumentException(
                'Invalid type of flag: ' . gettype($flag) . '. Only booleans are valid'
            );
        }
        $this->filters['no_subdealers'] = $flag;

        return $this;
    }

    public function includeRZData($flag = true)
    {
        $this->extra_fields[self::EXTRA_FIELDS_RZ] = (bool) $flag;

        return $this;
    }

    public function includeServices($flag = true)
    {
        $this->extra_fields[self::EXTRA_FIELDS_SERVICES] = (bool) $flag;

        return $this;
    }

    public function includeOpeningHours($flag = true)
    {
        $this->extra_fields[self::EXTRA_FIELDS_OPENING_HOURS] = (bool) $flag;

        return $this;
    }

    public function includeActivityCode($flag = true)
    {
        $this->extra_fields[self::EXTRA_FIELDS_ACTIVITY_CODE] = (bool) $flag;

        return $this;
    }

    public function includeRelatedEntities($flag = true)
    {
        $this->extra_fields[self::EXTRA_FIELDS_RELATED_ENTITIES] = (bool) $flag;

        return $this;
    }

    /**
     * @return Dealer[]
     */
    public function get()
    {
        $dealers = [];

        if ($extra_fields = array_filter($this->extra_fields)) {
            $this->dealer_loader->setExtraFields(array_keys($extra_fields));
            unset($extra_fields);
        }

        foreach ((array)json_decode($this->dealer_loader->fetch(), true) as $dealer_properties) {
            $dealer_properties = (array)$dealer_properties;
            if (!isset(
                $dealer_properties['code'],
                $dealer_properties['sitecode'],
                $dealer_properties['brand'],
                $dealer_properties['type']
            )) {
                if ($this->logger instanceof LoggerInterface) {
                    $this->logger->error('Fetched dealer data incomplete', ['data' => $dealer_properties]);
                }
                continue;
            }

            $dealer = new Dealer(
                $dealer_properties['code'],
                $dealer_properties['sitecode'],
                $dealer_properties['brand'],
                $dealer_properties['type']
            );

            $dealer->setProperties($dealer_properties);
            $dealer->setLoaded(true);
            $dealers[$dealer->getUID()] = $dealer;
        };

        return array_filter($dealers, function (Dealer $dealer) {
            return $this->filter($dealer);
        });
    }

    private function filter(Dealer $dealer)
    {
        if ($this->filters['brands'] and !in_array($dealer->getBrandId(), $this->filters['brands'])) {
            return false;
        }

        if ($this->filters['sincoms']) {
            $result = false;
            foreach ($this->filters['sincoms'] as $sincom) {
                if (preg_match('/^\d{7}$/', $sincom)) {
                    // Just SINCOM
                    if ($dealer->getSINCOM() === $sincom) {
                        $result = true;
                        break;
                    }
                } else {
                    // SINCOM.SITECODE
                    if ($dealer->getCode() === $sincom) {
                        $result = true;
                        break;
                    }
                }
            }

            if (!$result) {
                return false;
            }
            unset($result);
        }

        if ($this->filters['types'] and !in_array($dealer->getType(), $this->filters['types'])) {
            return false;
        }

        if ($this->filters['legal_entites'] and !in_array($dealer->getLegalEntity(), $this->filters['legal_entites'])) {
            return false;
        }

        if ($this->filters['maincodes'] and !in_array($dealer->getMainCode(), $this->filters['maincodes'])) {
            return false;
        }

        if ($this->filters['no_subdealers'] and $dealer->isSubDealer()) {
            return false;
        }

        if ($this->filters['oics'] and !in_array($dealer->getOIC(), $this->filters['oics'])) {
            return false;
        }

        return true;
    }
}
