<?php

namespace FCAPoland\DealerAPIHelper;

use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;
use Psr\SimpleCache\CacheInterface;

class DealerLoader
{
    /**
     * @var string Valid URL to API
     */
    private $dealer_api_url = 'https://api.fcapoland.pl/dealers';

    /**
     * @var array Any query params to be added when fetching dealers data (like 'extra_fields')
     */
    private $query_params = [];

    /**
     * @var LoggerInterface|NullLogger
     */
    private $logger;

    /**
     * @var resource
     */
    private $backup_resource;

    /**
     * @var int
     */
    private $backup_ttl;

    /**
     * @var CacheInterface
     */
    private $cache;

    /**
     * @var string
     */
    private $cache_key = 'fca_dealer_api_json';

    /**
     * @var int
     */
    private $cache_expiration;

    public function __construct(LoggerInterface $logger = null)
    {
        $this->logger = $logger ? $logger : new NullLogger();
    }

    public function fetch()
    {
        $dealers = $this->loadFromCache();
        if (is_null($dealers)) {
            if ($dealers = $this->loadFromAPI()) {
                $this->saveToCache($dealers);
            }
        }

        return $dealers;
    }

    private function loadFromCache()
    {
        if ($this->cache instanceof CacheInterface) {
            try {
                return $this->cache->get($this->getCacheKey());
            } catch (\Psr\SimpleCache\InvalidArgumentException $e) {
                $this->logger->error($e->getMessage());

                return null;
            }
        }

        return null;
    }

    private function saveToCache($data)
    {
        if ($this->cache instanceof CacheInterface) {
            try {
                $this->cache->set($this->getCacheKey(), $data, $this->cache_expiration);
            } catch (\Psr\SimpleCache\InvalidArgumentException $e) {
                $this->logger->error($e->getMessage());

                return;
            }
        }
    }

    private function loadFromAPI()
    {
        try {
            $curl_handle = curl_init();
            $url = $this->dealer_api_url;
            if ($this->query_params) {
                $url .= '?' . http_build_query($this->query_params);
            }
            curl_setopt($curl_handle, CURLOPT_URL, $url);
            curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($curl_handle, CURLOPT_HEADER, false);
            curl_setopt($curl_handle, CURLOPT_HTTPHEADER, ['Cache-Control: no-cache', 'Pragma: no-cache']);
            $dealers = curl_exec($curl_handle);
            curl_close($curl_handle);

            if (!$dealers) {
                $this->logger->error('Could not load dealers data directly from API');
                $this->logger->info('Restoring dealers data from backup...');
                $dealers = $this->loadFromBackup();
            } else {
                $this->backup($dealers);
            }
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
            $this->logger->info('Restoring dealers data from backup...');
            $dealers = $this->loadFromBackup();
        }

        return $dealers;
    }

    /**
     * @param string $url
     */
    public function setDealerAPIUrl($url)
    {
        $this->dealer_api_url = $url;
    }

    public function setCache(CacheInterface $cache, $ttl = null)
    {
        $ttl = intval($ttl);
        if ($ttl === 0) {
            // The default: 10 minutes
            $ttl = 10 * 60;
        }

        $this->cache_expiration = $ttl;
        $this->cache = $cache;
    }

    public function setBackup($resource, $ttl = null)
    {
        if (!is_resource($resource)) {
            throw new \InvalidArgumentException('Backup file is not a resource');
        }

        // Purposely or not: rewind the resource
        rewind($resource);

        $resource_meta_data = stream_get_meta_data($resource);
        if (in_array($resource_meta_data['mode'], ['r', 'w', 'a', 'x', 'c'])) { // all other modes support both: reading AND writing
            throw new \InvalidArgumentException('Backup file must be valid and readable and writeable');
        }
        if (!is_writable($resource_meta_data['uri']) or !is_readable($resource_meta_data['uri'])) {
            // check if the underlying data is readable AND writable
            throw new \InvalidArgumentException('Backup file must be valid and readable and writeable');
        }
        $this->backup_resource = $resource;

        $default = 43200; // 12 hours by default and when errors
        $ttl = intval($ttl);
        $this->backup_ttl = $ttl === 0 ? $default : $ttl;
    }

    private function loadFromBackup()
    {
        if (!$this->backup_resource) {
            $this->logger->warning('Backup resource undefined. Use `setBackup()`.');

            return '';
        }

        // Skip first line: it is backup creation date in format: YYYY-MM-DD hh:mm:ss
        if (false === fgets($this->backup_resource)) {
            // Empty file encountered
            $this->logger->warning('Backup resource is possibly empty.');

            return '';
        }

        // Read rest of the file as the proper backup data
        $backup = stream_get_contents($this->backup_resource);
        if (null === json_decode($backup)) {
            $this->logger->warning('Backup resource contains invalid data.');

            return '';
        }

        return $backup;
    }

    private function backup($dealers)
    {
        if (!$this->backup_resource) {
            $this->logger->warning('Backup resource undefined. Use `setBackup()`.');

            return;
        }

        // First line of the backup is the creation date in format: YYYY-MM-DD hh:mm:ss
        $creation_date = strtotime(fgets($this->backup_resource));

        if (time() - $creation_date < $this->backup_ttl) {
            // Too early to overwrite the backup
            return;
        }

        if (null === json_decode($dealers)) {
            $this->logger->warning('Received dealers data is invalid and thus will not be saved as backup');

            return;
        }

        rewind($this->backup_resource);
        ftruncate($this->backup_resource, 0);
        fwrite($this->backup_resource, date('Y-m-d H:i:s') . PHP_EOL);
        fwrite($this->backup_resource, (string)$dealers);
    }

    public function setExtraFields(array $extra_fields)
    {
        asort($extra_fields); // Sort the fields for the sake of cache key
        $this->query_params['extra_fields'] = join(',', array_filter($extra_fields));
    }

    public function clearCache()
    {
        $this->cache->clear();
    }

    private function getCacheKey()
    {
        if ($this->query_params) {
            $query_params = $this->query_params;
            asort($query_params); // Sort array to prevent duplicates (different cache keys pointing to the same contents)
            $suffix = md5(serialize($query_params));
        }

        return $this->cache_key . (isset($suffix) ? '_' . $suffix : '');
    }
}
