<?php

namespace Tests\Unit;

use FCAPoland\DealerAPIHelper\Dealer;
use FCAPoland\DealerAPIHelper\DealerLoader;
use FCAPoland\DealerAPIHelper\Dealers;

class DealersTest extends \PHPUnit_Framework_TestCase
{
    public function testGet()
    {
        $dealers = new Dealers($this->mockLoader());
        $result = $dealers->get();

        $this->assertInternalType('array', $result);

        $dealer_id = key($result);
        $this->assertRegExp('/^\d{7}\.\d{3}\.\d{2}\.(sales|aftersales)$/', $dealer_id);

        $dealer = array_pop($result);
        $this->assertInstanceOf(Dealer::class, $dealer);
    }

    private function mockLoader()
    {
        // Mock loader to fetch JSON from local file instead of calling external API
        $loader = $this->getMockBuilder(DealerLoader::class)
                       ->setMethods(['fetch'])
                       ->getMock();

        // This is fake dealers details that is used as a query result, including 'rzs' field
        $dealers_json = '[{"code":"0012345","sitecode":"001","brand":"00","name":"FOOBARMOTO SP. Z O.O.",' .
                        '"type":"sales","subdealer":false,"address":{"street":"FOOBAROWSKA 12","post_code":"12-345",' .
                        '"city":"Foobarowo","region":{"id":13,"code":"LDZ","name":"\u0142\u00f3dzkie"}},' .
                        '"coordinates":{"longitude":"19.494603","latitude":"51.90401"},"website":"http:\/\/www.foo-' .
                        'bar-moto.pl\/fiat?locationCode=00000123","email":"f12345s@dealer.fiat.pl","phones":' .
                        '["48123456789","48123456798"],"fax":"48123456799","level_of_service":null,' .
                        '"rz":{"last_name":"SalesRz","first_name":"FirstName","email":"firstname.sales@unknown.com",' .
                        '"phone":"512345678"},"rzs":{"sales":{"last_name":"SalesRz","first_name":"FirstName",' .
                        '"email":"firstname.sales@unknown.com","phone":"512345678"},"fcabank":' .
                        '{"last_name":"FcaBankRz","first_name":"FirstName","email":"firstname.fcabank@unknown.com",' .
                        '"phone":""},"leasys":{"last_name":"LeasysRz","first_name":"FirstName",' .
                        '"email":"firstname.leasysrz@somedomain.com","phone":""}}}]';
        $loader
            ->expects($this->atLeastOnce())
            ->method('fetch')
            ->willReturn($dealers_json);

        return $loader;
    }

    public function testWithSINCOMs()
    {
        $dealers = new Dealers();
        $sincoms = ['0075440.001'];
        $input = $sincoms;
        $dealers->withSINCOMs($input);
        $filters = $this->readAttribute($dealers, 'filters');
        $this->assertEquals($sincoms, $filters['sincoms']);

        $dealers = new Dealers();
        $sincoms = ['0075440.001', '0075366.000'];
        $input = $sincoms;
        $dealers->withSINCOMs($input);
        $filters = $this->readAttribute($dealers, 'filters');
        $this->assertEquals($sincoms, $filters['sincoms']);

        $dealers = new Dealers();
        $sincoms = ['0075373', '0075463'];
        $input = $sincoms;
        $dealers->withSINCOMs($input);
        $filters = $this->readAttribute($dealers, 'filters');
        $this->assertEquals($sincoms, $filters['sincoms']);

        $dealers = new Dealers();
        $sincoms = ['0075373', '0075463'];
        $input = $sincoms;
        $dealers
            ->withSINCOMs(['0012345'])
            ->withSINCOMs($input);
        $filters = $this->readAttribute($dealers, 'filters');
        $this->assertEquals($sincoms, $filters['sincoms']);
    }

    public function testWithSINCOMs_exception_1()
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withSINCOMs(['075463']);
    }

    public function testWithSINCOMs_exception_2()
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withSINCOMs(['0754634.00']);
    }

    public function testWithSINCOMs_exception_3()
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withSINCOMs(['0754634.001', '020202']);
    }

    public function testWithSINCOMs_exception_4()
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withSINCOMs(['']);
    }

    public function testWithSINCOMs_exception_5()
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withSINCOMs(['abcdefg']);
    }

    public function testWithBrands()
    {
        $dealers = new Dealers();
        $brands = ['00'];
        $input = $brands;
        $dealers->withBrands($input);
        $filters = $this->readAttribute($dealers, 'filters');
        $this->assertEquals($brands, $filters['brands']);

        $dealers = new Dealers();
        $brands = ['00', '57'];
        $input = $brands;
        $dealers->withBrands($input);
        $filters = $this->readAttribute($dealers, 'filters');
        $this->assertEquals($brands, $filters['brands']);

        $dealers = new Dealers();
        $input = ['00', 'jeep'];
        $brands = ['00', '57'];
        $dealers->withBrands($input);
        $filters = $this->readAttribute($dealers, 'filters');
        $this->assertEquals($brands, $filters['brands']);

        $dealers = new Dealers();
        $input = ['00', 'jeep'];
        $brands = ['00', '57'];
        $dealers
            ->withBrands(['83'])
            ->withBrands($input);
        $filters = $this->readAttribute($dealers, 'filters');
        $this->assertEquals($brands, $filters['brands']);
    }

    public function testWithBrands_exception_1()
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withBrands(['porsche']);
    }

    public function testWithBrands_exception_2()
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withBrands(['fiat', 'porsche']);
    }

    public function testWithBrands_exception_3()
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withBrands(['fiat', '990']);
    }

    public function testWithBrands_exception_4()
    {
        $this->expectException(\InvalidArgumentException::class);
        $dealers = new Dealers();
        $dealers->withBrands(['fiat', '']);
    }

    public function testIncludeRZData()
    {
        $dealers = new Dealers();
        $dealers->includeRZData();
        $extra_fields = $this->readAttribute($dealers, 'extra_fields');
        $this->assertTrue($extra_fields['rz']);
        $dealers->includeRZData(false);
        $extra_fields = $this->readAttribute($dealers, 'extra_fields');
        $this->assertFalse($extra_fields['rz']);
    }

    public function testIncludeServices()
    {
        $dealers = new Dealers();
        $dealers->includeServices();
        $extra_fields = $this->readAttribute($dealers, 'extra_fields');
        $this->assertTrue($extra_fields['services']);
        $dealers->includeServices(false);
        $extra_fields = $this->readAttribute($dealers, 'extra_fields');
        $this->assertFalse($extra_fields['services']);
    }

    public function testIncludeOpeningHours()
    {
        $dealers = new Dealers();
        $dealers->includeOpeningHours();
        $extra_fields = $this->readAttribute($dealers, 'extra_fields');
        $this->assertTrue($extra_fields['opening_hours']);
        $dealers->includeOpeningHours(false);
        $extra_fields = $this->readAttribute($dealers, 'extra_fields');
        $this->assertFalse($extra_fields['opening_hours']);
    }

    public function testIncludeActivityCode()
    {
        $dealers = new Dealers();
        $dealers->includeActivityCode();
        $extra_fields = $this->readAttribute($dealers, 'extra_fields');
        $this->assertTrue($extra_fields['activity_code']);
        $dealers->includeActivityCode(false);
        $extra_fields = $this->readAttribute($dealers, 'extra_fields');
        $this->assertFalse($extra_fields['activity_code']);
    }

    public function testIncludeRelatedEntities()
    {
        $dealers = new Dealers();
        $dealers->includeRelatedEntities();
        $extra_fields = $this->readAttribute($dealers, 'extra_fields');
        $this->assertTrue($extra_fields['related_entities']);
        $dealers->includeRelatedEntities(false);
        $extra_fields = $this->readAttribute($dealers, 'extra_fields');
        $this->assertFalse($extra_fields['related_entities']);
    }

    public function testGetRZs()
    {
        $dealers = new Dealers($this->mockLoader());
        $result = $dealers->get();

        /** @var Dealer $dealer */
        $dealer = array_pop($result);

        $rzs = $dealer->getRZs();

        $this->assertInternalType('array', $rzs);
        $this->assertNotEmpty($rzs);
        $this->assertArrayHasKey('sales', $rzs);
        $this->assertArrayHasKey('fcabank', $rzs);
        $this->assertArrayHasKey('leasys', $rzs);

        foreach (array('sales', 'fcabank', 'leasys') as $type) {
            $this->assertInternalType('array', $rzs[$type]);
            $this->assertNotEmpty($rzs[$type]);
            $this->assertArrayHasKey('first_name', $rzs[$type]);
            $this->assertNotEmpty($rzs[$type]['first_name']);
            $this->assertArrayHasKey('last_name', $rzs[$type]);
            $this->assertNotEmpty($rzs[$type]['last_name']);
            $this->assertArrayHasKey('email', $rzs[$type]);
            $this->assertNotEmpty($rzs[$type]['email']);
            $this->assertArrayHasKey('phone', $rzs[$type]);
        }
        unset($type);
    }
}
