<?php

namespace FCAPoland\FinancialAPI\RequestClient;

use Exception;
use FCAPoland\FinancialAPI\Config;
use FCAPoland\FinancialAPI\Model\CalculateResponseData;
use FCAPoland\FinancialAPI\Model\DetectResponseData;
use FCAPoland\FinancialAPI\Model\RequestData;
use FCAPoland\FinancialAPI\RequestClient;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\Request;

class HTTP implements RequestClient
{
    /**
     * @var Client
     */
    private $client;

    /**
     * @param Config $config
     */
    public function __construct(Config $config)
    {
        $this->client = new Client(['base_uri' => $config->getHttpApiUrl()]);
    }

    /**
     * @param RequestData $requestData
     * @return DetectResponseData
     * @throws GuzzleException
     * @throws Exception
     */
    public function detectStockCalculators(RequestData $requestData): DetectResponseData
    {
        $request = new Request('POST', 'detect-stock-calculations');
        $request->withAddedHeader('content-type', 'application/json');
        $response = $this->client->send($request, ['json' => $requestData->toArray()]);

        if ($response->getStatusCode() === 200 and $response->getHeader('Content-Type')[0] === 'application/json; charset=utf-8') {
            $responseArray = json_decode($response->getBody(), true);
            return new DetectResponseData($responseArray);
        } else {
            throw new Exception('Invalid detect calculator response');
        }
    }

    /**
     * @param RequestData $requestData
     * @param string $calculatorName
     * @return CalculateResponseData
     * @throws GuzzleException
     * @throws Exception
     */
    public function calculate(RequestData $requestData, string $calculatorName): CalculateResponseData
    {
        $request = new Request('POST', 'calculate/' . $calculatorName);
        $request->withAddedHeader('content-type', 'application/json');
        $response = $this->client->send($request, ['json' => $requestData->toArray(), 'query' => ['withSolData' => (int) $requestData->isWithSolData()]]);

        if ($response->getStatusCode() === 200 and $response->getHeader('Content-Type')[0] === 'application/json; charset=utf-8') {
            $responseArray = json_decode($response->getBody(), true);
            return new CalculateResponseData($responseArray);
        } else {
            throw new Exception('Invalid detect calculator response');
        }
    }
}
