<?php

namespace FCAPoland\FinancialAPI\RequestClient;

use Exception;
use FCAPoland\FinancialAPI\Config;
use FCAPoland\FinancialAPI\Model\CalculateResponseData;
use FCAPoland\FinancialAPI\Model\DetectResponseData;
use FCAPoland\FinancialAPI\Model\RequestData;
use FCAPoland\FinancialAPI\RequestClient;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\Request;

class HTTP implements RequestClient
{
    const DETECTION_URL = 'calculation/detect-website-products/:BRAND/:WEBSITE_ID/:ENV';
    const CALCULATE_URL = 'calculation/calculate/:BRAND';

    /**
     * @var Client
     */
    private $client;
    /**
     * @var Config
     */
    private $config;

    /**
     * @param Config $config
     */
    public function __construct(Config $config)
    {
        $this->client = new Client(['base_uri' => $config->getHttpApiUrl()]);
        $this->config = $config;
    }

    /**
     * @param RequestData $requestData
     * @return DetectResponseData
     * @throws GuzzleException
     * @throws Exception
     */
    public function detectCalculators(RequestData $requestData): DetectResponseData
    {
        $request = new Request('POST', $this->getDetectionUrl($requestData));
        $request->withAddedHeader('content-type', 'application/json');
        $response = $this->client->send($request, ['json' => $requestData->toArray(),
            'query' => [
                'withSolData' => (int)$requestData->isWithSolData(),
                'withCalculationParams' => (int)$requestData->isWithCalculationParams()]
        ]);

        if ($response->getStatusCode() === 200 and $response->getHeader('Content-Type')[0] === 'application/json; charset=utf-8') {
            $responseArray = json_decode($response->getBody(), true);
            return new DetectResponseData($responseArray);
        } else {
            throw new Exception('Invalid detect calculator response');
        }
    }

    /**
     * @param RequestData $requestData
     * @return CalculateResponseData
     * @throws GuzzleException
     */
    public function calculate(RequestData $requestData): CalculateResponseData
    {
        $request = new Request('POST', $this->getCalculateUrl($requestData));
        $request->withAddedHeader('content-type', 'application/json');

        $response = $this->client->send(
            $request,
            [
                'json' => $requestData->toArray(),
                'query' => [
                    'withSolData' => (int)$requestData->isWithSolData(),
                    'withCalculationParams' => (int)$requestData->isWithCalculationParams()]
            ]
        );

        if ($response->getStatusCode() === 200 and $response->getHeader('Content-Type')[0] === 'application/json; charset=utf-8') {
            $responseArray = json_decode($response->getBody(), true);
            return new CalculateResponseData($responseArray);
        } else {
            throw new Exception('Invalid detect calculator response');
        }
    }

    private function getDetectionUrl(RequestData $requestData): string
    {
        return strtr(self::DETECTION_URL, [
            ':BRAND' => $requestData->getBrand(),
            ':WEBSITE_ID' => $this->config->getWebsiteId(),
            ':ENV' => $this->config->getEnvironment(),
        ]);
    }

    private function getCalculateUrl(RequestData $requestData): string
    {
        return strtr(self::CALCULATE_URL, [
            ':BRAND' => $requestData->getBrand(),
        ]);
    }
}
