<?php

namespace FCAPoland\FinancialAPI\Model;

use Exception;
use FCAPoland\FinancialAPI\Model\Request\ExtraServices;
use FCAPoland\FinancialAPI\Model\Request\Fields;
use FCAPoland\FinancialAPI\Model\Request\Vehicle;

class RequestData
{
    const ACTION_CALCULATE = 'calculate';
    const ACTION_DETECT_PRODUCTS = 'detect-website-products';

    private $data = [];

    private $withSolData = false;

    private $withCalculationParams = false;

    private $brand;

    public function __construct(?string $brand = null)
    {
        $this->brand = $brand;
    }

    /**
     * @param string $vehicleId
     * @param Vehicle $vehicle
     * @param Fields|null $fields
     * @param int|null $calculatorId
     * @param array $extraServices
     * @return $this
     * @throws Exception
     */
    public function addVehicle(
        string $vehicleId,
        Vehicle $vehicle,
        ?Fields $fields = null,
        ?int $calculatorId = null,
        array $extraServices = []
    ): self {
        if (isset($this->data[$vehicleId])) {
            throw new Exception('Object with the same vehicleId already exists (' . $vehicleId . ')');
        }

        $this->data[$vehicleId] = [
            'vehicle' => $vehicle,
            'fields' => $fields,
            'extraServices' => new ExtraServices($extraServices)
        ];

        if ($calculatorId !== null) {
            $this->data[$vehicleId]['calculatorId'] = $calculatorId;
        }

        return $this;
    }

    /**
     * @return array
     */
    public function getVehicles(): array
    {
        return $this->data;
    }

    /**
     * @param string $vehicleId
     * @return Vehicle|null
     */
    public function getVehicle(string $vehicleId): ?Vehicle
    {
        return $this->data[$vehicleId] ?? null;
    }

    /**
     * @param string $vehicleId
     * @param int $calculatorId
     * @return $this
     */
    public function updateVehicleCalculatorId(string $vehicleId, int $calculatorId): self
    {
        $this->data[$vehicleId]['calculatorId'] = $calculatorId;
        return $this;
    }

    /**
     * @param bool $withSolData
     */
    public function setWithSolData(bool $withSolData): void
    {
        $this->withSolData = $withSolData;
    }

    /**
     * @return bool
     */
    public function isWithSolData(): bool
    {
        return $this->withSolData;
    }

    public function setWithCalculationParams(bool $withCalculationParams): void
    {
        $this->withCalculationParams = $withCalculationParams;
    }

    public function isWithCalculationParams(): bool
    {
        return $this->withCalculationParams;
    }

    /**
     * @return ?string
     */
    public function getBrand(): ?string
    {
        return $this->brand;
    }

    /**
     * @param ?string $brand
     */
    public function setBrand(?string $brand): void
    {
        $this->brand = $brand;
    }

    /**
     * @return array
     */
    public function toArray(): array
    {
        $array = [];
        foreach ($this->data as $key => $value) {
            if (isset($value['fields'])) {
                $array[$key] = [
                    'vehicle' => $value['vehicle']->toArray(),
                    'fields' => $value['fields']->toArray(),
                    'extraServices' => $value['extraServices']->getextraServices(),
                    'calculatorId' => $value['calculatorId'] ?? null
                ];
            } else {
                $array[$key] = [
                    'vehicle' => $value['vehicle']->toArray(),
                    'extraServices' => $value['extraServices']->getextraServices(),
                    'calculatorId' => $value['calculatorId'] ?? null
                ];
            }
        }

        return $array;
    }

    /**
     * @param string $action
     * @param string $websiteId
     * @param bool $isProd
     * @return array
     */
    public function toArrayForWebSocket(string $action, string $websiteId, bool $isProd): array
    {
        if ($action === self::ACTION_CALCULATE) {
            $array = [
                'action' => $action,
                'withSolData' => $this->isWithSolData(),
                'withCalculationParams' => $this->isWithCalculationParams(),
                'singleMessageResponse' => true,
                'data' => []
            ];
        } else {
            $array = [
                'action' => $action,
                'websiteId' => $websiteId,
                'isProduction' => $isProd,
                'withSolData' => $this->isWithSolData(),
                'withCalculationParams' => $this->isWithCalculationParams(),
                'singleMessageResponse' => true,
                'data' => []
            ];
            if ($this->getBrand() != null) {
                $array['brand'] = $this->getBrand();
            }
        }

        foreach ($this->data as $key => $value) {
            if ($action === self::ACTION_CALCULATE && empty($value['calculatorId'] ?? null)) {
                continue;
            }

            if (isset($value['fields'])) {
                $array['data'][$key] = [
                    'vehicle' => $value['vehicle']->toArray(),
                    'fields' => $value['fields']->toArray(),
                    'calculatorId' => $value['calculatorId'] ?? null,
                    'extraServices' => $value['extraServices']->getextraServices(),
                ];
            } else {
                $array['data'][$key] = [
                    'vehicle' => $value['vehicle']->toArray(),
                    'calculatorId' => $value['calculatorId'] ?? null,
                    'extraServices' => $value['extraServices']->getextraServices(),
                ];
            }
        }

        return $array;
    }

    public function clearVehicles()
    {
        $this->data = [];
    }
}
