<?php

namespace FCAPoland\FinancialAPI\RequestClient;

use FCAPoland\FinancialAPI\Config;
use FCAPoland\FinancialAPI\Exception\WebSocketRequestDisabled;
use FCAPoland\FinancialAPI\Exception\WebSocketTooLongResponse;
use FCAPoland\FinancialAPI\Model\CalculateResponseData;
use FCAPoland\FinancialAPI\Model\DetectResponseData;
use FCAPoland\FinancialAPI\Model\RequestData;
use FCAPoland\FinancialAPI\RequestClient;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;
use WebSocket\Client;
use WebSocket\Message\Message;
use WebSocket\TimeoutException;

class WebSocket implements RequestClient
{
    use LoggerAwareTrait;

    /**
     * @var Config
     */
    private $config;

    /**
     * @var Client
     */
    private $wsClient;

    public function __construct(Config $config, ?LoggerInterface $logger)
    {
        $this->config = $config;
        $this->setLogger($logger ?? new NullLogger());
        if ($this->config->isWebsocketEnable()) {
            $this->wsClient = new Client(
                $this->config->getWebsocketApiUrl(),
                [
                    'persistent' => true,
                    'return_obj' => true,
                    'logger' => $this->logger ?? new NullLogger(),
                    'timeout' => 10,
                ]
            );
            // Dla inicjacji połączenia
            $this->wsClient->ping();
        }
    }

    public function detectCalculators(RequestData $requestData): DetectResponseData
    {
        if (!$this->config->isWebsocketEnable()) {
            throw new WebSocketRequestDisabled();
        }

        $this->wsClient->send(json_encode($requestData->toArrayForWebSocket(
            RequestData::ACTION_DETECT_PRODUCTS,
            $this->config->getWebsiteId(),
            $this->config->getEnvironment() != 'staging'
        )));

        try {
            $response = $this->wsClient->receive();

            if ($response instanceof Message && $response->hasContent() && mb_strlen($response->getContent()) > 2) {
                $responseArray = json_decode($response->getContent(), true);
                return new DetectResponseData($responseArray);
            } else {
                throw new \Exception('Invalid detect calculator response');
            }
        } catch (TimeoutException $e) {
            throw new WebSocketTooLongResponse('Invalid detect calculator response - too long wait for WS answer');
        }
    }

    public function calculate(RequestData $requestData): CalculateResponseData
    {
        if (!$this->config->isWebsocketEnable()) {
            throw new WebSocketRequestDisabled();
        }

        $this->wsClient->send(json_encode($requestData->toArrayForWebSocket(
            RequestData::ACTION_CALCULATE,
            $this->config->getWebsiteId(),
            $this->config->getEnvironment() != 'staging'
        )));

        try {
            $response = $this->wsClient->receive();

            if ($response instanceof Message && $response->hasContent() && mb_strlen($response->getContent()) > 2) {
                $responseArray = json_decode($response->getContent(), true);
                return new CalculateResponseData($responseArray);
            } else {
                throw new \Exception('Invalid detect calculator response');
            }
        } catch (TimeoutException $e) {
            throw new WebSocketTooLongResponse('Invalid detect calculator response - too long wait for WS answer');
        }
    }


    public function isEnabled(): bool
    {
        return $this->config->isWebsocketEnable();
    }

    public function getWsClient(): Client
    {
        return $this->wsClient;
    }
}
