<?php

namespace FCAPoland\FinancialAPI\RequestClient;

use Exception;
use FCAPoland\FinancialAPI\Config;
use FCAPoland\FinancialAPI\Model\CalculateResponseData;
use FCAPoland\FinancialAPI\Model\DetectResponseData;
use FCAPoland\FinancialAPI\Model\RequestData;
use FCAPoland\FinancialAPI\RequestClient;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Psr7\Request;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;

class HTTP implements RequestClient
{
    use LoggerAwareTrait;
    public const DETECTION_URL = 'calculation/detect-website-products/:BRAND/:WEBSITE_ID/:ENV';
    public const DETECTION_URL_NO_BRAND = 'calculation/detect-website-products/:WEBSITE_ID/:ENV';
    public const CALCULATE_URL = 'calculation/calculate';

    /**
     * @var Client
     */
    private $client;
    /**
     * @var Config
     */
    private $config;

    public function __construct(Config $config, ?LoggerInterface $logger)
    {
        $this->client = new Client(['base_uri' => $config->getHttpApiUrl()]);
        $this->config = $config;
        $this->setLogger($logger ?? new NullLogger());
    }

    /**
     * @throws GuzzleException
     * @throws Exception
     */
    public function detectCalculators(RequestData $requestData): DetectResponseData
    {
        $request = new Request('POST', $this->getDetectionUrl($requestData));
        $request->withAddedHeader('content-type', 'application/json');

        $response = $this->client->send($request, ['json' => $requestData->toArray(),
            'query' => [
                'withSolData' => (int)$requestData->isWithSolData(),
                'withCalculationParams' => (int)$requestData->isWithCalculationParams()]
        ]);

        if ($response->getStatusCode() === 200 and
            $response->getHeader('Content-Type')[0] === 'application/json; charset=utf-8'
        ) {
            $responseArray = json_decode($response->getBody(), true);
            return new DetectResponseData($responseArray);
        } else {
            throw new Exception('Invalid detect calculator response');
        }
    }

    /**
     * @throws GuzzleException
     */
    public function calculate(RequestData $requestData): CalculateResponseData
    {
        $request = new Request('POST', self::CALCULATE_URL);
        $request->withAddedHeader('content-type', 'application/json');

        $response = $this->client->send(
            $request,
            [
                'json' => $requestData->toArray(),
                'query' => [
                    'withSolData' => (int)$requestData->isWithSolData(),
                    'withCalculationParams' => (int)$requestData->isWithCalculationParams()]
            ]
        );

        if ($response->getStatusCode() === 200 and
            $response->getHeader('Content-Type')[0] === 'application/json; charset=utf-8'
        ) {
            $responseArray = json_decode($response->getBody(), true);
            return new CalculateResponseData($responseArray);
        } else {
            throw new Exception('Invalid detect calculator response');
        }
    }

    private function getDetectionUrl(RequestData $requestData): string
    {
        return strtr(
            $requestData->getBrand() !== null ? self::DETECTION_URL : self::DETECTION_URL_NO_BRAND,
            [
                ':BRAND' => $requestData->getBrand(),
                ':WEBSITE_ID' => $this->config->getWebsiteId(),
                ':ENV' => $this->config->getEnvironment(),
            ]
        );
    }
}
