<?php

namespace FCAPoland\FormsLib\Resource\Dealer;

use FCAPoland\FormsLib\Exception\FetchDealersFCAAPIException;
use FCAPoland\FormsLib\Exception\FormsLIBException;
use FCAPoland\FormsLib\Form\Form;
use FCAPoland\FormsLib\Resource\Dealer;
use Psr\SimpleCache\CacheInterface;
use Psr\SimpleCache\InvalidArgumentException;

/**
 * Class Collection
 * @package FCAPoland\FormsLib\Resource\Dealer
 */
class Collection
{
    /** @var string string */
    public static $user_agent = 'FCA FORMS LIB';

    const CACHE_KEY = 'formsLib.dealers.:BRAND:';

    const FCA_API_DEALER_URL = 'https://api.fcapoland.pl/dealers?types=sales&brands=:BRAND:';

    /** @var string  */
    private $brand_id;

    /** @var CacheInterface */
    private $cache;

    /** @var Form */
    private $formObject;

    /** @var array */
    private $dealers = [];

    /**
     * Dealer constructor.
     * @param Form $form
     */
    public function __construct(Form $form)
    {
        $this->brand_id = $form->getBrandID();
        $this->formObject = $form;
        if ($this->formObject->getCache() instanceof CacheInterface) {
            $this->cache = $this->formObject->getCache();
        }
    }

    /**
     * @return array
     */
    public function getDealers()
    {
        return$this->dealers;
    }

    /**
     * @return $this
     * @throws FormsLIBException
     */
    public function initDealers()
    {
        try {
            if ($this->cache instanceof CacheInterface) {
                // z cache
                $dealers_json = $this->cache->get($this->getCacheKey(), false);
                if (!$dealers_json) {
                    // Gdy nie ma dealerów w cache pobranie z api
                    $dealers_json = $this->fetch();
                    // Zapis nowo pobranych dealerów do BC cache
                    $this->saveDealerToBackupCache($dealers_json);
                }
            } else {
                // pobranie dealerów z api
                $dealers_json = $this->fetch();
                // Zapis nowo pobranych dealerów do BC cache
                $this->saveDealerToBackupCache($dealers_json);
            }
        } catch (FetchDealersFCAAPIException $e) {
            $dealers_json = $this->loadDealerFromBackupCache();
        } catch (InvalidArgumentException $e) {
            throw new FormsLIBException($e->getMessage(), null, $e);
        }

        if (!$this->checkDealersJSON($dealers_json)) {
            throw new FormsLIBException('Dealers data is incorrect. Can not works without dealers.');
        }

        $dealers_as_array = json_decode($dealers_json, true);

        foreach ($dealers_as_array as $dealer) {
            if ($this->formObject->withDealers() and is_array($this->formObject->getDealersList())) {
                if (!in_array($dealer['code'], $this->formObject->getDealersList())) {
                    continue;
                }
            }
            $dealer_for_save = new Dealer($dealer['code'], $dealer['sitecode']);
            $dealer_for_save->setProperties($dealer);

            $this->dealers[$dealer['code'] . ':' . $dealer['sitecode']] = $dealer_for_save;
        }
        return $this;
    }

    /**
     * @param string $json
     * @return bool
     */
    private function checkDealersJSON(string $json)
    {
        json_decode($json);
        return (json_last_error() == JSON_ERROR_NONE);
    }

    /**
     * @return false|string
     */
    private function loadDealerFromBackupCache()
    {
        $dealers_json = file_get_contents($this->getBackupCachePath());
        return $dealers_json;
    }

    /**
     * @param $dealers_json
     * @return bool
     */
    private function saveDealerToBackupCache($dealers_json)
    {
        $file_e = file_exists($this->getBackupCachePath());
        $file_save_status = file_put_contents($this->getBackupCachePath(), $dealers_json, LOCK_EX);
        if (!$file_e) {
            chmod($this->getBackupCachePath(), 0666);
        }
        return $file_save_status;
    }

    /**
     * @return string
     */
    private function getFCAAPIDealerURL()
    {
        return str_replace(':BRAND:', $this->brand_id, self::FCA_API_DEALER_URL);
    }

    /**
     * @return string
     */
    private function getCacheKey()
    {
        return str_replace(':BRAND:', $this->brand_id, self::CACHE_KEY);
    }

    /**
     * @return string
     */
    private function getBackupCachePath()
    {
        return $this->formObject->getHelpersCachePath() . $this->getCacheKey();
    }

    /**
     * @return bool|string
     * @throws FetchDealersFCAAPIException
     */
    private function fetch()
    {
        try {
            $curl_handle = curl_init();
            curl_setopt($curl_handle, CURLOPT_URL, $this->getFCAAPIDealerURL());
            curl_setopt($curl_handle, CURLOPT_CONNECTTIMEOUT, 15);
            curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl_handle, CURLOPT_USERAGENT, Collection::$user_agent);
            $query = curl_exec($curl_handle);
            $httpcode = curl_getinfo($curl_handle, CURLINFO_HTTP_CODE);
            curl_close($curl_handle);

            if ($query === false or $httpcode >= 400) {
                throw new FetchDealersFCAAPIException('Unable to fetch dealers data from FCA Dealer API');
            }

            if (!$this->checkDealersJSON($query)) {
                throw new FetchDealersFCAAPIException('Dealers data fetched from FCA Dealer API is incorrect!');
            }

            return $query;
        } catch (\Exception $e) {
            throw new FetchDealersFCAAPIException(
                'Unable to fetch dealers data from FCA Dealer API. ERROR:' . $e->getMessage()
            );
        }
    }
}
