<?php

namespace FCAPoland\FormsLib\Form;

use FCAPoland\ApiPrivacyHelper\Disclaimer;
use FCAPoland\ApiPrivacyHelper\Exception;
use FCAPoland\FormsLib\ClientParam\ClientParamInterface;
use FCAPoland\FormsLib\ClientParam\SessionClientParam;
use FCAPoland\FormsLib\Config\FCAAPIAuth;
use FCAPoland\FormsLib\Exception\FieldInitException;
use FCAPoland\FormsLib\Exception\FormsLIBException;
use FCAPoland\FormsLib\Exception\InvalidArgumentException;
use FCAPoland\FormsLib\Field\CommunicationChannelField;
use FCAPoland\FormsLib\Field\EmailField;
use FCAPoland\FormsLib\Field\FieldInterface;
use FCAPoland\FormsLib\Field\PhoneField;
use FCAPoland\FormsLib\Field\RecaptchaField;
use FCAPoland\FormsLib\Field\RepeatEmailField;
use FCAPoland\FormsLib\Field\Rule\RuleInterface;
use FCAPoland\FormsLib\Resource\Dealer;
use FCAPoland\FormsLib\Resource\Dealer\Collection;
use FCAPoland\LeadsAPIHelper\Lead;
use FCAPoland\LeadsAPIHelper\Request\Create;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;
use ReflectionClass;

/**
 * Class Form
 * @package FCAPoland\FormsLib
 */
class Form extends BaseForm
{
    protected $helpers_cache_path = './forms_lib';

    /**
     * @param array $form
     * @throws Exception
     * @throws InvalidArgumentException
     * @throws \Psr\SimpleCache\InvalidArgumentException
     * @throws FieldInitException
     * @throws FormsLIBException
     * @throws \FCAPoland\LeadsAPIHelper\InvalidArgumentException
     */
    public function initForm(array $form)
    {
        if (!isset($form['id']) or
            !isset($form['enabled']) or
            !isset($form['with_subdealers']) or
            !isset($form['disclaimer_id']) or
            !isset($form['cta']) or
            !isset($form['brand_id']) or
            !isset($form['cid']) or
            !isset($form['source']) or
            !isset($form['fields'])
        ) {
            throw new InvalidArgumentException('Incorrect form config');
        }

        if (!is_array($form['cta'])) {
            throw new InvalidArgumentException('Incorrect form config - CTA must be array');
        }

        if ($this->cache instanceof CacheInterface) {
            $this->masterkey->setCache($this->cache);
        }

        if ($this->logger instanceof LoggerInterface) {
            $this->masterkey->setLogger($this->logger);
        }

        $this->setFormID($form['id']);
        $this->setEnable((bool) $form['enabled']);
        $this->setDisclaimerID($form['disclaimer_id']);

        $this->setCTA($form['cta']);

        $this->setBrandID($form['brand_id']);
        $this->setCid($form['cid']);
        $this->setSource($form['source']);

        if (isset($form['oid']) and is_int($form['oid'])) {
            $this->setOID($form['oid']);
        }

        if (isset($form['with_dealers'])
            and $form['with_dealers'] == true
        ) {
            $this->setWithDealers(true);
        }
        if (isset($form['with_subdealers'])
            and $form['with_subdealers'] == false
        ) {
            $this->setWithSubDealers(false);
        }
        if (isset($form['dealers_list'])
             and is_array($form['dealers_list'])
        ) {
            $this->setDealersList($form['dealers_list']);
        }

        $this->masterkey->setBrand($this->getBrandID());
        $this->masterkey->setAppCid($this->getCID());
        if ($this->getOID()) {
            $this->masterkey->setAppOid($this->getOID());
        }
        $this->masterkey->setAppSource($this->getSource());


        if (!is_array($form['fields'])) {
            throw new InvalidArgumentException("Incorrect field config for {$this->getFormID()}");
        }
        $this->initFieldsObjects($form['fields']);

        $this->initDisclaimer();

        $this->initDealers();
    }

    /**
     * @return array
     */
    public function getFormArray()
    {
        return [
            'id' => $this->getFormID(),
            'cta' => $this->getCTA(),
            'disclaimer_id' => $this->getDisclaimer()->getDisclaimerID(),
            'fields' => $this->getFormFieldsAsArray(true),
            'agreements' => $this->getDisclaimer()->getAgreements()->getFormElements(),
            'required_fields' => $this->getRequiredFields(),
            'disclaimer' => [
                'information'   => [
                    'keys' => $this->getDisclaimer()->getInformation()->getInformationKeys(),
                    'content' => $this->getDisclaimer()->getInformation()->getInformations(),
                ],
            ],
            'form_dealers' => $this->getDealersFormArray()
        ];
    }

    /**
     * @param bool $with_keys
     * @return array
     */
    private function getFormFieldsAsArray($with_keys = false)
    {
        /** @var FieldInterface $field */
        $fields = [];
        foreach ($this->fields as $field) {
            $rules = [];
            $fieldStorage = [];
            /**
             * @var  $rule
             * @var  $ruleObject RuleInterface
             */
            foreach ($field->getRuleObjects() as $rule => $ruleObject) {
                $rules[] = [
                    'rule' => $ruleObject->getRuleName(),
                    'regex' => $ruleObject->getRegex(),
                    'message' => $ruleObject->getMessage()
                ];
            }

            if (is_array($field->getOptionValues())) {
                $optionValues = [];
                foreach ($field->getOptionValues() as $k_optionValue => $optionValue) {
                    $optionValues[] = [
                        'value' => $k_optionValue,
                        'label' => $optionValue
                    ];
                }
                $fieldStorage['input'] = $optionValues;
            }

            $fieldStorage['name'] = $field->getName();
            $fieldStorage['value'] = $field->getValue();
            $fieldStorage['type'] = $field->getType();
            $fieldStorage['label'] = $field->getLabel();
            $fieldStorage['format'] = $field->getFormat();
            $fieldStorage['placeholder'] = $field->getPlaceHolder();
            $fieldStorage['required'] = $field->isRequired();
            $fieldStorage['rules'] = $rules;

            if ($field->getFieldName() == 'RecaptchaField') {
                /** @var $field RecaptchaField */
                $fieldStorage['recaptcha'] = [
                    'site_key' => $field->getSiteKey(),
                ];
            }

            if ($with_keys) {
                $fields[$field->getName()] = $fieldStorage;
            } else {
                $fields[] = $fieldStorage;
            }
        }
        return $fields;
    }

    /**
     * @return array
     */
    public function getRequiredFields()
    {
        $required_fields = [];

        /** @var FieldInterface $field */
        foreach ($this->getFields() as $field) {
            if ($field->isRequired()) {
                $required_fields[] = $field->getName();
            }
        }

        return array_merge(
            $required_fields, // Wymagane pola form
            $this->disclaimer->getAgreements()->requiredFields() // wymagane pola zgód
        );
    }

    /**
     * @param array $fields
     * @throws FieldInitException
     */
    private function initFieldsObjects(array $fields)
    {
        foreach ($fields as $field_object => $field_params) {
            try {
                $objectReflection = new ReflectionClass($field_object);
                /** @var FieldInterface $object */
                $object = $objectReflection->newInstanceArgs($field_params);
                $object->initRules();
                $this->fields[$object->getName()] = $object;
            } catch (\ReflectionException $e) {
                throw new FieldInitException("Can not field init " . $field_object);
            }
        }
    }

    /**
     * @throws Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     * @throws \FCAPoland\ApiPrivacyHelper\Exception
     */
    protected function initDisclaimer()
    {
        $this->disclaimer = new Disclaimer($this->getDisclaimerID(), $this->getHelpersCachePath());

        if ($this->cache instanceof CacheInterface) {
            $this->disclaimer->setCache($this->cache);
        }
        $this->disclaimer->getDisclaimer();
    }

    /**
     * @return $this
     * @throws FormsLIBException
     */
    private function initDealers()
    {
        if ($this->withDealers()) {
            $dealers = new Collection($this);
            $dealers->setWithSubDealers($this->withSubDealers());
            $dealers->initDealers();
            $this->form_dealers = $dealers;
        }
        return $this;
    }

    /**
     * @return array|bool
     */
    public function getDealersFormArray()
    {
        $dealers = [];
        if ($this->withDealers()) {
            /** @var Dealer $dealer */
            foreach ($this->form_dealers->getDealers() as $dealer) {
                $dealers[$dealer->getCode()] = [
                    'sincom' => $dealer->getCode(),
                    'name' => $dealer->getName(),
                    'address' => $dealer->getAddress(),
                    'coordinates' => $dealer->getCoordinates(),
                    'phones' => $dealer->getPhones(),
                    'email' => $dealer->getEmail()
                ];
            }
            return count($dealers) > 0 ? $dealers : false;
        }
        return false;
    }


    /**
     * Sprawdzenie pół form
     */
    private function checkFormData()
    {
        /** @var FieldInterface $field */
        foreach ($this->fields as $field) {
            if ($field instanceof RepeatEmailField) {
                if (!$field->isValid($this->form_data->getPropertyByField($field)) or
                    !$field->isValidRepeatEmail(
                        $this->form_data->getPropertyByField($field),
                        $this->form_data->getPropertyByField(new EmailField())
                    )
                ) {
                    $this->form_data_errors[$field->getName()] = $field->getErrorsMessages();
                }
            } else {
                if ($field instanceof PhoneField) {
                    if (
                        isset($this->fields['communication_channel']) and
                        $this->form_data->getPropertyByField($this->fields['communication_channel']) == Lead::COMMUNICATION_CHANNEL_MAIL
                    ) {
                        if (!empty($this->form_data->getPropertyByField($this->fields['phone']))) {
                            if (!$field->isValid($this->form_data->getPropertyByField($field))) {
                                $this->form_data_errors[$field->getName()] = $field->getErrorsMessages();
                            }
                        }
                    } else {
                        if (!$field->isValid($this->form_data->getPropertyByField($field))) {
                            $this->form_data_errors[$field->getName()] = $field->getErrorsMessages();
                        }
                    }
                } else {
                    if (!$field->isValid($this->form_data->getPropertyByField($field))) {
                        $this->form_data_errors[$field->getName()] = $field->getErrorsMessages();
                    }
                }
            }
        }
    }

    /**
     * @param array $form_data
     */
    public function setFormData(array $form_data)
    {
        parent::setFormData($form_data);
        $this->checkRequiredFields();
        $this->checkFormData();
    }

    /**
     * Sprawdzenie czy wszystkie wymagane pola są w przesłanych danych form
     */
    private function checkRequiredFields()
    {
        foreach ($this->getRequiredFields() as $requiredField) {
            if (
                isset($this->fields['communication_channel']) and
                $this->form_data->getPropertyByField($this->fields['communication_channel']) == Lead::COMMUNICATION_CHANNEL_MAIL
                and $requiredField === 'phone'
            ) {
                continue;
            }
            if (!$this->form_data->hasProperty($requiredField)) {
                $this->form_data_absent_fields[] = $requiredField;
            }
        }
    }

    /**
     * @param FCAAPIAuth $auth_config
     * @param bool $debug_saved_data
     * @return array
     * @throws FormsLIBException
     * @throws \FCAPoland\LeadsAPIHelper\Exception
     * @throws \FCAPoland\LeadsAPIHelper\InvalidArgumentException
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function submitForm(FCAAPIAuth $auth_config, $debug_saved_data = false)
    {
        if (!$this->isFormDataValid()) {
            throw new FormsLIBException('Can not submit not valid form');
        }

        if (!($this->client_params instanceof ClientParamInterface)) {
            $this->client_params = new SessionClientParam();
        }

        if ($this->client_params->getCID()) {
            $this->masterkey->setCid($this->client_params->getCID());
        }
        if ($this->client_params->getOID()) {
            $this->masterkey->setOid($this->client_params->getOID());
        }
        if ($this->client_params->getSource()) {
            $this->masterkey->setSource($this->client_params->getSource());
        }

        $lead = new Lead();

        if ($this->form_data->getBrandID() !== null) {
            $lead->setBrandId($this->form_data->getBrandID());
            $this->masterkey->setBrand($this->form_data->getBrandID());
        } else {
            $lead->setBrandId($this->getBrandID());
        }

        $lead->setDisclaimerId($this->getDisclaimerID());

        if ($this->form_data->getCTA() != null) {
            $lead->setType($this->form_data->getCTA());
        } else {
            $lead->setType($this->getCTA()[0]);
        }

        $this->masterkey->find($lead->getType());
        $master_key = $this->masterkey->getMasterkey();

        $lead->setMasterkey($master_key);

        if ($this->form_data->getFirstName()) {
            $lead->setFirstName($this->form_data->getFirstName());
        }
        if ($this->form_data->getLastName()) {
            $lead->setLastName($this->form_data->getLastName());
        }
        if ($this->form_data->getCompany()) {
            $lead->setCompany($this->form_data->getCompany());
        }
        if ($this->form_data->getAddress()) {
            $lead->setAddress($this->form_data->getAddress());
        }
        if ($this->form_data->getZipCode()) {
            $lead->setZipCode($this->form_data->getZipCode());
        }
        if ($this->form_data->getCity()) {
            $lead->setCity($this->form_data->getCity());
        }
        if ($this->form_data->getEmail()) {
            $lead->setEmail($this->form_data->getEmail());
        }
        if ($this->form_data->getPhone()) {
            $lead->setPhoneNumber($this->form_data->getPhone());
        }
        if ($this->form_data->getDealerSincom()) {
            $lead->setDealerSincom($this->form_data->getDealerSincom());
        }
        if ($this->form_data->getModelID()) {
            $lead->setModelId($this->form_data->getModelID());
        }
        if ($this->form_data->getAppointmentDate()) {
            $lead->setAppointmentDate($this->form_data->getAppointmentDate());
        }
        if ($this->form_data->getRelatedID()) {
            $lead->setRelatedId($this->form_data->getRelatedID());
        }
        if ($this->form_data->getOrigin()) {
            // Magiczne dodanie parametrów z query string do origin'a
            // lepszego miejsca nie znalazłem

            // Pobranie parametróœ zapisanych w sesji
            $client_session_params = $this->getClientParams()->getAllParams();
            $client_session_params = array_filter($client_session_params);
            // Dostarczony origin
            $origin = $this->form_data->getOrigin();
            // Sprawdzenie czy dostarczony origin jest prawidłowym url
            if (filter_var($origin, FILTER_VALIDATE_URL) !== false) {
                // Sytuacja gdy url ma ogonek - wyciągamy tylko ogonek jako string
                if ($query_string = parse_url($origin, PHP_URL_QUERY)) {
                    // ogonek z stringa zamieniamy na tablice
                    parse_str($query_string, $query_string_as_array);
                    // Łączymy parametry z stringa z dostarczonymi z sesji
                    $client_session_params = array_merge($query_string_as_array, $client_session_params);
                    // Budujemy url (usuwamy query z orginalnego origin'a) i doklejamy wszystkie inne parametry
                    $origin = str_replace($query_string, '', $origin) . http_build_query($client_session_params);
                } else {
                    // URl nie ma ogona dodajemy tylko parametry z sesji
                    $origin = $origin .'?'. http_build_query($client_session_params);
                }
            } else {
                // Gdy origin nie jest url - np sztuczne generowany origin - dodajemy tylko parametry z sesji
                $origin = $origin .'?'. http_build_query($client_session_params);
            }
            $lead->setOrigin($origin);
        }
        if ($this->form_data->getUserAgent()) {
            $lead->setUserAgent($this->form_data->getUserAgent());
        }
        if ($this->form_data->getDevice()) {
            $lead->setDevice($this->form_data->getDevice());
        }
        if ($this->form_data->getIP()) {
            $lead->setIp($this->form_data->getIP());
        }
        if ($this->form_data->getComment()) {
            $lead->setComment($this->form_data->getComment());
        }

        if ($this->form_data->getCommunicationChannel()) {
            $lead->setCommunicationChannel($this->form_data->getCommunicationChannel());
        }

        $map_agreement_fields = $this->getDisclaimer()->getAgreements()->agreementsMapper();

        foreach ($map_agreement_fields as $map_agreement_field => $map_agreement_field_privacy) {
            foreach ($map_agreement_field_privacy as $privacy_key) {
                if ($this->form_data->getAgreementByKey($map_agreement_field) == 1) {
                    $lead->setPrivacy($privacy_key, Lead::PRIVACY_ACCEPTED);
                } else if ($this->form_data->getAgreementByKey($map_agreement_field) == 0) {
                    $lead->setPrivacy($privacy_key, Lead::PRIVACY_REFUSED);
                } else {
                    $lead->setPrivacy($privacy_key, Lead::PRIVACY_UNKNOWN);
                }
            }
        }

        $api_request = new Create(
            $auth_config->getApiUrl(),
            $auth_config->getApiUser(),
            $auth_config->getApiPass()
        );

        $api_request->setLead($lead);

        try {
            $api_request->execute();
        } catch (\FCAPoland\LeadsAPIHelper\Exception $exception) {
            return [
                'status'       => 'error',
                'message'       => $exception->getMessage(),
            ];
        }

        $return_data = [
        'status'       => 'success',
        'lead_id'      => $lead->getId(),
        'cta'          => $lead->getType(),
        'brand'        => $lead->getBrandId(),
        'masterkey_id' => $master_key->getId(),
        ];

        if ($debug_saved_data) {
            $return_data['lead_data'] = $lead->getProperties();
            $return_data['masterkey'] = [
               'id' => $master_key->getId(),
               'brand' => $master_key->getBrand(),
               'cta' => $master_key->getType(),
               'cid' => $master_key->getCid(),
               'oid' => $master_key->getOid(),
               'source' => $master_key->getSource(),
            ];
        }

        return $return_data;
    }
}
