<?php

namespace FCAPoland\FormsLib;

use FCAPoland\ApiPrivacyHelper\Exception;
use FCAPoland\ApiPrivacyHelper\PrivacyPolicy;
use FCAPoland\FormsLib\Config\LibConfig;
use FCAPoland\FormsLib\Exception\FormNotFoundException;
use FCAPoland\FormsLib\Exception\FormsNotFoundException;
use FCAPoland\FormsLib\Exception\InvalidArgumentException;
use FCAPoland\FormsLib\Form\Form;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;

/**
 * Class FormManager
 * @package FCAPoland\FormsLib
 */
class FormManager
{

    /** @var LibConfig */
    private $lib_config;

    /** @var CacheInterface */
    private $cache;

    /** @var  LoggerInterface */
    private $logger;

    /** @var array */
    private $forms;

    /**
     * @param LibConfig $lib_config
     * @throws FormsNotFoundException
     * @throws \FCAPoland\ApiPrivacyHelper\Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function setLibConfig(LibConfig $lib_config)
    {
        $this->lib_config = $lib_config;
        $this->loadForms();
    }

    /**
     * @param CacheInterface $cache
     */
    public function setCache(CacheInterface $cache)
    {
        $this->cache = $cache;
    }

    /**
     * @param LoggerInterface $logger
     */
    public function setLogger(LoggerInterface $logger)
    {
        $this->logger = $logger;
    }

    /**
     * @param $id
     * @return Form
     * @throws FormNotFoundException
     */
    public function getFormByID($id)
    {
        if (!isset($this->forms[$id])) {
            throw new FormNotFoundException("Form {$id} not found");
        }
        if (!$this->forms[$id]->isEnable()) {
            throw new FormNotFoundException("Form {$id} is disabled");
        }
        return $this->forms[$id];
    }

    /**
     * @throws Exception
     * @throws FormsNotFoundException
     * @throws \Psr\SimpleCache\InvalidArgumentException
     * @throws \Exception
     */
    private function loadForms()
    {
        $this->forms = [];
        if ($this->lib_config->getForms() and is_array($this->lib_config->getForms())) {
            foreach ($this->lib_config->getForms() as $form_config) {
                try {
                    $form = new Form();
                    if ($this->cache instanceof CacheInterface) {
                        $form->setCache($this->cache);
                    }
                    if ($this->logger instanceof LoggerInterface) {
                        $form->setLogger($this->logger);
                    }
                    $form->setHelpersCachePath($this->lib_config->getStoragePath() . '_form_');
                    $form->initForm($form_config);
                    $this->forms[$form->getFormID()] = $form;
                } catch (InvalidArgumentException $e) {
                    throw new \Exception($e->getMessage());
                }
            }
        } else {
            throw new FormsNotFoundException('Forms not found. Please Check lib config');
        }
    }

    /**
     * @return string|null
     * @throws Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function getPrivacyPolicy()
    {
        try {
            $privacy_policy = new PrivacyPolicy($this->lib_config->getStoragePath());
            $privacy_policy->setPrivacyPolicyHash($this->lib_config->getPrivacyPolicyHash());
            if ($this->cache instanceof CacheInterface) {
                $privacy_policy->setCache($this->cache);
            }
        } catch (Exception $e) {
            if ($this->logger instanceof LoggerInterface) {
                $this->logger->error('Can not init PrivacyPolicy object: ' . $e->getMessage());
            }
        }
        return isset($privacy_policy) ? $privacy_policy->getPrivacyPolicyContent() : null;
    }
}
