<?php

namespace FCAPoland\FormsLib\Field;

use FCAPoland\FormsLib\Exception\FieldRuleNotFound;
use FCAPoland\FormsLib\Field\Rule\RuleInterface;
use ReflectionClass;

/**
 * Class BaseField
 * @package FCAPoland\FormsLib\Field
 */
class BaseField implements FieldInterface
{
    const TYPE_HIDDEN = 'hidden';
    const TYPE_TEXT = 'text';
    const TYPE_TEXTAREA = 'textarea';
    const TYPE_EMAIL = 'email';
    const TYPE_DATE = 'date';
    const TYPE_PHONE = 'phone';
    const TYPE_SELECT = 'select';
    const TYPE_RADIO = 'radio';
    const TYPE_CHECKBOX = 'checkbox';
    const TYPE_NUMBER = 'number';


    /**
     * Rules for validation
     * @var array
     */
    protected $rules;

    /**
     * Rules for validation
     * @var array
     */
    protected $rule_objects;

    /**
     * Type of form field
     * @var string
     */
    protected $type;

    /**
     * Name tag of field
     * @var string
     */
    protected $name;

    /**
     * Label of form field
     * @var string|null
     */
    protected $label;

    /**
     * Place holder of form field
     * @var string|null
     */
    protected $place_holder;

    /**
     * Required of form field
     * @var bool
     */
    protected $required = true;

    /**
     * Format of form field
     * @var string|null
     */
    protected $format;

    /**
     * Options for select,radio[],checkbox[]
     * @var array|null
     */
    protected $option_values;

    /** @var array|null */
    protected $error_messages;

    /**
     * Set default rules for this field
     */
    protected function setDefaultRules()
    {
    }

    /**
     * Check if value is valid
     * @param $value
     * @return bool
     */
    public function isValid($value)
    {
        /** @var RuleInterface $rule */
        foreach ($this->getRuleObjects() as $rule) {
            if (!$rule->isValidValue($value)) {
                $this->error_messages[] = $rule->getMessage();
            }
        }
        return $this->error_messages == null ? true : false;
    }

    /**
     * Get rules array
     * @return mixed
     */
    public function getRules()
    {
        return $this->rules;
    }

    /**
     * @return array
     */
    public function getRuleObjects()
    {
        return $this->rule_objects;
    }

    /**
     * @param array $rule_objects
     * @return BaseField
     */
    public function setRuleObjects(array $rule_objects): BaseField
    {
        $this->rule_objects = $rule_objects;
        return $this;
    }

    /**
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * @return string|null
     */
    public function getLabel()
    {
        return $this->label;
    }

    /**
     * @return string|null
     */
    public function getPlaceHolder()
    {
        return $this->place_holder;
    }

    /**
     * @return bool
     */
    public function isRequired()
    {
        return $this->required;
    }

    /**
     * @return string|null
     */
    public function getFormat()
    {
        return $this->format;
    }

    /**
     * @return array|null
     */
    public function getOptionValues()
    {
        return $this->option_values;
    }

    /**
     * Rules setter
     * @param $rules
     */
    public function setRules($rules)
    {
        $this->rules = $rules;
    }

    /**
     * @param $type
     * @return void
     */
    public function setType($type)
    {
        $this->type = $type;
    }

    /**
     * @param $name
     * @return void
     */
    public function setName($name)
    {
        $this->name = $name;
    }

    /**
     * @param $label
     * @return void
     */
    public function setLabel($label)
    {
        $this->label = $label;
    }

    /**
     * @param $place_holder
     * @return void
     */
    public function setPlaceHolder($place_holder)
    {
        $this->place_holder = $place_holder;
    }

    /**
     * @param $required
     * @return void
     */
    public function setRequired($required)
    {
        $this->required = (bool) $required;
    }

    /**
     * @param $format
     * @return void
     */
    public function setFormat($format)
    {
        $this->format = $format;
    }

    /**
     * @param $options
     * @return void
     */
    public function setOptionValues($options)
    {
        $this->option_values = $options;
    }

    /**
     * Init rules as object.
     * @throws FieldRuleNotFound
     */
    public function initRules()
    {
        foreach ($this->getRules() as $rule_object => $rule_params) {
            $rule_params = array_merge([$this], $rule_params);
            try {
                $objectReflection = new ReflectionClass($rule_object);
                /** @var RuleInterface $object */
                $object = $objectReflection->newInstanceArgs($rule_params);
                $this->rule_objects[$rule_object] = $object;
            } catch (\ReflectionException $e) {
                throw new FieldRuleNotFound("Do not found rule " . get_class($rule_params));
            }
        }
    }

    /**
     * @return mixed
     */
    public function getFieldName()
    {
        $path = explode('\\', get_class($this));
        return array_pop($path);
    }

    /**
     * @return array|null
     */
    public function getErrorsMessages()
    {
        return $this->error_messages;
    }
}
