<?php


namespace Unit;


use FCAPoland\FormsLib\Field\BaseField;
use FCAPoland\FormsLib\Field\FieldInterface;
use FCAPoland\FormsLib\Field\Rule\BaseRule;
use FCAPoland\FormsLib\Field\Rule\CheckRecaptcha;
use FCAPoland\FormsLib\Field\Rule\InArray;
use FCAPoland\FormsLib\Field\Rule\IsEmail;
use FCAPoland\FormsLib\Field\Rule\IsPhone;
use FCAPoland\FormsLib\Field\Rule\MaxLength;
use FCAPoland\FormsLib\Field\Rule\MinLength;
use FCAPoland\FormsLib\Field\Rule\NotEmpty;
use FCAPoland\FormsLib\Field\Rule\RuleInterface;
use FCAPoland\FormsLib\Field\Rule\ValidDate;
use FCAPoland\FormsLib\Field\Rule\ValidDealerSincom;
use FCAPoland\FormsLib\Field\Rule\ValidPostCode;
use PHPUnit\Framework\TestCase;

/**
 * Class RulesTest
 * @package Unit
 */
class RulesTest extends TestCase
{
    public function testBaseRule()
    {
        $object = new BaseRule(new BaseField());
        $this->assertInstanceOf(RuleInterface::class, $object);
        $object->setRegex('(.*)');
        $this->assertEquals('(.*)', $object->getRegex());
        $this->assertTrue($object->isValidValue('A'));
        $object->setRegex('/TEST\-(.*)/');
        $this->assertTrue($object->isValidValue('TEST-A'));
        $this->assertTrue($object->isValidValue('TEST-B'));
        $this->assertFalse($object->isValidValue('1TEST'));
        $this->assertEquals(BaseRule::class, $object->getRuleName());
        $this->assertInstanceOf(FieldInterface::class, $object->getField());
        $this->assertEquals(BaseField::class, $object->getFieldName());

       // $this->expectException(TypeError::class);
        $this->assertFalse($object->isValidValue([]));
    }

    public function testCheckRecaptcha()
    {
        $object = new CheckRecaptcha(new BaseField(), 'site_key', 'secret_key');
        $this->assertInstanceOf(RuleInterface::class, $object);
        $this->assertEquals('site_key', $object->getSiteKey());
        $this->assertEquals('secret_key', $object->getSecretKey());
        // TODO do przerobienia aby nie łączyło się z zew serwerami
        // Sprawdzane jest w google czy kod jest prawidłowy dla fake danych
        $this->assertFalse($object->isValidValue('xxx'));
        // TODO dorobienie poprawnej walidacji - LVL_UP!
        // Czyli jakaś symulacja recaptch'y
    }

    public function testInArray()
    {
        $object = new InArray(new BaseField(), ['111']);
        $this->assertInstanceOf(RuleInterface::class, $object);
        $this->assertFalse($object->isValidValue('222'));
        $this->assertTrue($object->isValidValue(111));
        $this->assertTrue($object->isValidValue('111'));
        $this->assertFalse($object->isValidValue(['111']));
    }

    public function testIsEmail()
    {
        $object = new IsEmail(new BaseField());
        $this->assertInstanceOf(RuleInterface::class, $object);
        $this->assertFalse($object->isValidValue('222'));
        $this->assertTrue($object->isValidValue('test@test.com'));
        $this->assertTrue($object->isValidValue('test@test.com.pl'));
        $this->assertTrue($object->isValidValue('test.test@test.com.pl'));
    }

    public function testIsPhone()
    {
        $object = new IsPhone(new BaseField());
        $this->assertInstanceOf(RuleInterface::class, $object);
        $this->assertFalse($object->isValidValue('222'));
        $this->assertFalse($object->isValidValue('6666666666'));
        $this->assertFalse($object->isValidValue('+44 666-666-666'));
        $this->assertTrue($object->isValidValue('666-666-666'));
        $this->assertTrue($object->isValidValue('666666666'));
        $this->assertTrue($object->isValidValue('+48 666-666-666'));
        $this->assertTrue($object->isValidValue('+48666666666'));
    }

    public function testMaxLength()
    {
        $object = new MaxLength(new BaseField(), 2);
        $this->assertInstanceOf(RuleInterface::class, $object);
        $this->assertFalse($object->isValidValue('222'));
        $this->assertFalse($object->isValidValue('6666666666'));
        $this->assertFalse($object->isValidValue('abc'));
        $this->assertTrue($object->isValidValue(''));
        $this->assertTrue($object->isValidValue('1'));
        $this->assertTrue($object->isValidValue('ab'));
    }

    public function testMinLength()
    {
        $object = new MinLength(new BaseField(), 2);
        $this->assertInstanceOf(RuleInterface::class, $object);
        $this->assertFalse($object->isValidValue('1'));
        $this->assertFalse($object->isValidValue(''));
        $this->assertTrue($object->isValidValue('ab'));
        $this->assertTrue($object->isValidValue('aaa'));
    }

    public function testNoEmpty()
    {
        $object = new NotEmpty(new BaseField());
        $this->assertInstanceOf(RuleInterface::class, $object);
        $this->assertFalse($object->isValidValue(''));
        $this->assertFalse($object->isValidValue(null));
        $this->assertFalse($object->isValidValue([]));
        $this->assertTrue($object->isValidValue('0'));
        $this->assertTrue($object->isValidValue('ab'));
        $this->assertTrue($object->isValidValue('aaa'));
    }

    public function testValidDate()
    {
        $object = new ValidDate(new BaseField());
        $this->assertInstanceOf(RuleInterface::class, $object);
        $this->assertFalse($object->isValidValue('99-12-2017'));
        $this->assertFalse($object->isValidValue('12-99-2017'));
        $this->assertFalse($object->isValidValue('12-99-18'));
        $this->assertFalse($object->isValidValue('2020-01-01'));
        $this->assertTrue($object->isValidValue('02-12-2020'));
        $this->assertTrue($object->isValidValue('05-01-2020'));
    }

    public function testDealerSincom()
    {
        $object = new ValidDealerSincom(new BaseField());
        $this->assertInstanceOf(RuleInterface::class, $object);
        $this->assertFalse($object->isValidValue(''));
        $this->assertFalse($object->isValidValue('666666666'));
        $this->assertFalse($object->isValidValue('6666666.5666'));
        $this->assertTrue($object->isValidValue('1111111.000'));
    }

    public function testPostCode()
    {
        $object = new ValidPostCode(new BaseField());
        $this->assertInstanceOf(RuleInterface::class, $object);
        $this->assertFalse($object->isValidValue(''));
        $this->assertFalse($object->isValidValue('83000'));
        $this->assertFalse($object->isValidValue('83 000'));
        $this->assertTrue($object->isValidValue('83-000'));
    }
}