<?php
declare(strict_types=1);

namespace FCAPoland\FormsLib\Resource;

/**
 * Class Dealer
 * @package FCAPoland\FormsLib\Resource
 */
class Dealer
{
    /**
     * List of object's properties that can be directly returned when invoking `asArray()` method
     *
     * @var array
     */
    protected $properties = array();

    /**
     * @var 7-digit (including leading zeroes) dealer's main code
     */
    private $sincom;

    /**
     * @var 3-digit (including leading zeroes) dealer's site code
     */
    private $sitecode;


    /**
     * Resource_Dealer constructor.
     *
     * @param                            $sincom
     * @param                            $sitecode
     *
     * @internal param array $dealer_data
     */
    public function __construct($sincom, $sitecode)
    {
        $this->sincom = $sincom;
        $this->sitecode = $sitecode;
    }

    /**
     * @param $dealer_data
     * @return Dealer
     */
    public function setProperties($dealer_data)
    {
        $this->properties = $dealer_data;
        return $this;
    }

    /**
     * Get property like 'code', 'sitecode', 'brand', 'name', 'type', 'address', 'website', 'email', 'phones', etc.
     *
     * @param $name
     *
     * @return mixed|null
     */
    private function getProperty($name)
    {
        return array_key_exists($name, $this->properties) ? $this->properties[$name] : null;
    }

    /**
     * Return dealer code in form: SINCOM.SITECODE
     *
     * @return string
     */
    public function getCode()
    {
        if ($sincom = $this->getSINCOM() and $sitecode = $this->getSiteCode()) {
            return $sincom . '.' . $sitecode;
        }
        return null;
    }

    /**
     * UID (Unique ID) is a composition of:
     *  - SINCOM (7 digits)
     *  - Site code (3 digits)
     *  - brand ID (2 digits)
     *  - service type ("sales" or "aftersales"
     *
     * This function returns this values glued (by default) with dot separator, for example:
     *
     *     0075450.001.00.sales
     *
     * @param string $separator
     *
     * @return string
     */
    public function getUID($separator = '.')
    {
        if ($sincom = $this->getSINCOM()
            and $sitecode = $this->getSiteCode()
            and $brand_id = $this->getBrandId()
            and $type = $this->getProperty('type')
        ) {
            return join($separator, array($sincom, $sitecode, $brand_id, $type));
        }
        return null;
    }

    /**
     * @return mixed|null
     */
    public function getSINCOM()
    {
        return $this->getProperty('code');
    }

    /**
     * @return mixed|null
     */
    public function getLegalEntity()
    {
        return $this->getProperty('legal_entity');
    }

    /**
     * @return mixed|null
     */
    public function getSiteCode()
    {
        return $this->getProperty('sitecode');
    }

    /**
     * @return mixed|null
     */
    public function getBrandId()
    {
        return $this->getProperty('brand');
    }

    /**
     * @return mixed|null
     */
    public function getName()
    {
        return $this->getProperty('name');
    }

    /**
     * @return mixed|null
     */
    public function getEmail()
    {
        return $this->getProperty('email');
    }

    /**
     * @return mixed|null
     */
    public function getAddress()
    {
        return $this->getProperty('address');
    }

    /**
     * @return bool|mixed
     */
    public function getAddressStreet()
    {
        $address = (array) $this->getAddress();

        return array_key_exists('street', $address) ? $address['street'] : false;
    }

    /**
     * @return bool|mixed
     */
    public function getAddressCity()
    {
        $address = (array) $this->getAddress();

        return array_key_exists('city', $address) ? $address['city'] : false;
    }

    /**
     * @return bool|mixed
     */
    public function getAddressPostCode()
    {
        $address = (array) $this->getAddress();

        return array_key_exists('post_code', $address) ? $address['post_code'] : false;
    }

    /**
     * @return bool|mixed
     */
    public function getAddressRegion()
    {
        $address = (array) $this->getAddress();

        return array_key_exists('region', $address) ? $address['region'] : false;
    }

    /**
     * @return bool|mixed
     */
    public function getAddressRegionId()
    {
        $region = (array) $this->getAddressRegion();

        return array_key_exists('id', $region) ? $region['id'] : false;
    }

    /**
     * @return bool|mixed
     */
    public function getAddressRegionName()
    {
        $region = (array) $this->getAddressRegion();

        return array_key_exists('name', $region) ? $region['name'] : false;
    }

    /**
     * @return array
     */
    public function getPhones()
    {
        return (array) $this->getProperty('phones');
    }

    /**
     * Gets single available phone number
     *
     * @return bool|mixed
     */
    public function getPhone()
    {
        if ($phones = $this->getPhones()) {
            return isset($phones[0]) ? $phones[0] : false;
        }
        return false;
    }

    /**
     * Gets single available phone number for freespee
     *
     * @return bool|mixed
     */
    public function getPhoneFreespee()
    {
        if ($phones = $this->getPhones()) {
            $phone = $this->getPhone();
            $phone = str_replace(array( ' ', '-' ), array( '', '' ), $phone);
            if (strlen($phone) < 9) {
                return false;
            }

            return substr($phone, - 9);
        }

        return false;
    }

    /**
     * @return mixed|null
     */
    public function getFax()
    {
        return $this->getProperty('fax');
    }

    /**
     * @return bool
     */
    public function hasSalesService()
    {
        $services = (array) $this->getProperty('services');

        return array_key_exists('sales', $services) and $services['sales'];
    }

    /**
     * @return mixed|null
     */
    public function isSubDealer()
    {
        return $this->getProperty('subdealer');
    }

    /**
     * @return mixed|null
     */
    public function getRZ()
    {
        return $this->getProperty('rz');
    }

    /**
     * @return mixed|null
     */
    public function getRZFirstName()
    {
        $rz = (array) $this->getRZ();

        return array_key_exists('first_name', $rz) ? $rz['first_name'] : null;
    }

    /**
     * @return mixed|null
     */
    public function getRZLastName()
    {
        $rz = (array) $this->getRZ();

        return array_key_exists('last_name', $rz) ? $rz['last_name'] : null;
    }

    /**
     * @return mixed|null
     */
    public function getRZEmail()
    {
        $rz = (array) $this->getRZ();

        return array_key_exists('email', $rz) ? $rz['email'] : null;
    }

    /**
     * @return mixed|null
     */
    public function getRZPhone()
    {
        $rz = (array) $this->getRZ();

        return array_key_exists('phone', $rz) ? $rz['phone'] : null;
    }

    /**
     * @param null $type
     * @return mixed|null
     */
    public function getCoordinates($type = null)
    {
        if ($type === null) {
            return $this->getProperty('coordinates');
        }

        $coordinates = (array) $this->getCoordinates();

        if (stripos($type, 'long') !== false) {
            return isset($coordinates['longitude']) ? $coordinates['longitude'] : null;
        }

        if (stripos($type, 'lat') !== false) {
            return isset($coordinates['latitude']) ? $coordinates['latitude'] : null;
        }
        return null;
    }
}
