<?php
declare(strict_types=1);

namespace FCAPoland\FormsLib;

use Exception;
use FCAPoland\ApiPrivacyHelper\StorageInterface;

/**
 * Class Storage
 * @package FCAPoland\FormsLib
 */
class Storage implements StorageInterface
{
    const STORAGE_FILE = 'helpers_cache/privacy_helper_storage';

    /** @var array */
    private $storage = [];

    /** @var string */
    private $storage_file;

    /**
     * Storage constructor.
     * @param null $storage_file
     * @throws Exception
     */
    public function __construct($storage_file = null)
    {
        if ($storage_file === null) {
            $this->storage_file = self::STORAGE_FILE;
        } else {
            $this->storage_file = $storage_file;
        }
        if (!file_exists($this->storage_file)) {
            $this->createStorageFile();
        }
        $this->readStorage();
    }

    /**
     * @param $id
     * @param $data
     * @throws Exception
     */
    public function saveDisclaimer(int $id, $data)
    {
        $this->readStorage();
        $this->storage['disclaimer'][$id] = $data;
        $this->saveStorage();
    }

    /**
     * @param $id
     * @return mixed
     * @throws Exception
     */
    public function getDisclaimer(int $id)
    {
        $this->readStorage();
        return $this->storage['disclaimer'][$id] ?? false;
    }

    /**
     * @param $hash
     * @param $data
     * @throws Exception
     */
    public function savePrivacyPolicy(string $hash, $data)
    {
        $this->readStorage();
        $this->storage['privacy_policy'][$hash] = $data;
        $this->saveStorage();
    }

    /**
     * @param $hash
     * @return mixed
     * @throws Exception
     */
    public function getPrivacyPolicy(string $hash)
    {
        $this->readStorage();
        return $this->storage['privacy_policy'][$hash] ?? false;
    }

    /**
     * @throws Exception
     */
    private function readStorage()
    {
        $storage_file = file_get_contents($this->storage_file);
        if ($storage_file === false) {
            throw new Exception('Can not read storage file');
        }
        if (strlen($storage_file) == 0) {
            $this->storage = [];
        }
        $this->storage = json_decode($storage_file, true);
    }

    /**
     * @param string $key
     * @param $data
     * @throws Exception
     */
    public function saveToStorage(string $key, $data)
    {
        $this->readStorage();
        $this->storage[$key] = $data;
        $this->saveStorage();
    }

    /**
     * @param string $key
     * @return mixed|bool
     * @throws Exception
     */
    public function readFromStorage(string $key)
    {
        $this->readStorage();
        return $this->storage[$key] ?? false;
    }

    /**
     * Save to storage file
     */
    private function saveStorage()
    {
        if (!is_array($this->storage)) {
            $this->storage = [];
        }
        $json_storage = json_encode($this->storage);
        file_put_contents($this->storage_file, $json_storage, LOCK_EX);
    }

    /**
     * @param string $form_and_brand_id
     * @param $data
     * @throws Exception
     */
    public function saveDealers(string $form_and_brand_id, $data)
    {
        $this->readStorage();
        $this->storage['dealers'][$form_and_brand_id] = $data;
        $this->saveStorage();
    }

    /**
     * @param $form_and_brand_id
     * @return mixed
     */
    public function getDealers(string $form_and_brand_id)
    {
        return $this->storage['dealers'][$form_and_brand_id] ?? false;
    }

    /**
     * Create storage file if not exists
     */
    private function createStorageFile()
    {
        if (!file_exists($this->storage_file)) {
            $storage_file = fopen($this->storage_file, 'w');
            fclose($storage_file);
            chmod($this->storage_file, 0664);
        }
    }

    /**
     * Remove storage file and clear storage
     */
    public function removeStorage()
    {
        if (file_exists($this->storage_file)) {
            unlink($this->storage_file);
            $this->storage = [];
        }
    }
}
