<?php
declare(strict_types=1);

namespace FCAPoland\FormsLib\Field;

use FCAPoland\FormsLib\Exception\FieldRuleNotFound;
use FCAPoland\FormsLib\Field\Rule\RuleInterface;
use ReflectionClass;
use ReflectionException;

/**
 * Class BaseField
 * @package FCAPoland\FormsLib\Field
 */
class BaseField implements FieldInterface
{
    const TYPE_HIDDEN = 'hidden';
    const TYPE_TEXT = 'text';
    const TYPE_TEXTAREA = 'textarea';
    const TYPE_EMAIL = 'email';
    const TYPE_DATE = 'date';
    const TYPE_PHONE = 'phone';
    const TYPE_SELECT = 'select';
    const TYPE_RADIO = 'radio';
    const TYPE_CHECKBOX = 'checkbox';
    const TYPE_NUMBER = 'number';
    const TYPE_RECAPTCHA = 'recaptcha';


    /**
     * Rules for validation
     * @var array
     */
    protected $rules = [];

    /**
     * Rules for validation
     * @var array
     */
    protected $rule_objects = [];

    /**
     * Type of form field
     * @var string
     */
    protected $type;

    /**
     * Name tag of field
     * @var string
     */
    protected $name;

    /**
     * Label of form field
     * @var string|null
     */
    protected $label;

    /**
     * Place holder of form field
     * @var string|null
     */
    protected $place_holder;

    /**
     * Required of form field
     * @var bool
     */
    protected $required = true;

    /**
     * Format of form field
     * @var string|null
     */
    protected $format;

    /**
     * ?????????
     * @var mixed
     */
    protected $value;

    /**
     * Options for select,radio[],checkbox[]
     * @var array|null
     */
    protected $option_values;

    /** @var array */
    protected $error_messages = [];

    /**
     * @return mixed
     */
    public function getValue()
    {
        return $this->value;
    }

    /**
     * @param mixed $value
     */
    public function setValue($value): void
    {
        $this->value = $value;
    }

    /**
     * Set default rules for this field
     */
    protected function setDefaultRules()
    {
    }

    /**
     * Check if value is valid
     * @param $value
     * @return bool
     */
    public function isValid($value): bool
    {
        // Reset tablicy z błędami, dla każdej z walidowanych wartości powinnien być resetowany
        $this->error_messages = [];
        foreach ($this->getRuleObjects() as $rule) {
            if (!$rule->isValidValue($value)) {
                $this->error_messages[] = $rule->getMessage();
            }
        }
        return count($this->error_messages) == 0;
    }


    /**
     * Get rules array
     * @return mixed
     */
    public function getRules(): array
    {
        return $this->rules;
    }

    /**
     * @return RuleInterface[]
     */
    public function getRuleObjects(): array
    {
        return $this->rule_objects;
    }

    /**
     * @param RuleInterface[] $rule_objects
     * @return BaseField
     */
    public function setRuleObjects(array $rule_objects): BaseField
    {
        $this->rule_objects = $rule_objects;
        return $this;
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @return string|null
     */
    public function getLabel(): ?string
    {
        return $this->label;
    }

    /**
     * @return string|null
     */
    public function getPlaceHolder(): ?string
    {
        return $this->place_holder;
    }

    /**
     * @return bool
     */
    public function isRequired(): bool
    {
        return $this->required;
    }

    /**
     * @return string|null
     */
    public function getFormat(): ?string
    {
        return $this->format;
    }

    /**
     * @return array|null
     */
    public function getOptionValues(): ?array
    {
        return $this->option_values;
    }

    /**
     * Rules setter
     * @param array $rules
     */
    public function setRules(array $rules)
    {
        $this->rules = $rules;
    }

    /**
     * @param string $type
     * @return void
     */
    public function setType(string $type)
    {
        $this->type = $type;
    }

    /**
     * @param string $name
     * @return void
     */
    public function setName(string $name)
    {
        $this->name = $name;
    }

    /**
     * @param string $label
     * @return void
     */
    public function setLabel(string $label)
    {
        $this->label = $label;
    }

    /**
     * @param string|null $place_holder
     * @return void
     */
    public function setPlaceHolder(?string  $place_holder)
    {
        $this->place_holder = $place_holder;
    }

    /**
     * @param bool $required
     * @return void
     */
    public function setRequired(bool $required)
    {
        $this->required = $required;
    }

    /**
     * @param string|null $format
     * @return void
     */
    public function setFormat(?string $format)
    {
        $this->format = $format;
    }

    /**
     * @param array $options
     * @return void
     */
    public function setOptionValues(array $options)
    {
        $this->option_values = $options;
    }

    /**
     * Init rules as object.
     * @throws FieldRuleNotFound
     */
    public function initRules()
    {
        // Reset obiektów zasad
        $this->rule_objects = [];
        foreach ($this->getRules() as $rule_object => $rule_params) {
            $rule_params = array_merge([$this], $rule_params);
            try {
                $objectReflection = new ReflectionClass($rule_object);
                /** @var RuleInterface $object */
                $object = $objectReflection->newInstanceArgs($rule_params);
                $this->rule_objects[$rule_object] = $object;
            } catch (ReflectionException $e) {
                throw new FieldRuleNotFound("Do not found rule " . get_class($rule_params));
            }
        }
    }

    /**
     * @return string
     */
    public function getFieldName(): string
    {
        return get_class($this);
    }

    /**
     * @return array
     */
    public function getErrorsMessages(): array
    {
        return $this->error_messages;
    }
}
