<?php
declare(strict_types=1);

namespace FCAPoland\FormsLib\Form;

use FCAPoland\ApiPrivacyHelper\Disclaimer;
use FCAPoland\ApiPrivacyHelper\StorageInterface;
use FCAPoland\FormsLib\ClientParam\ClientParamInterface;
use FCAPoland\FormsLib\Resource\Dealer\Collection;
use FCAPoland\LeadsAPIHelper\MasterkeyFinder;
use Psr\Log\LoggerAwareTrait;
use Psr\SimpleCache\CacheInterface;

/**
 * Class BaseForm
 * @package FCAPoland\FormsLib
 */
abstract class BaseForm
{
    use LoggerAwareTrait;

    /**
     * @var string
     */
    protected $formID;

    /** @var bool  */
    protected $enable = true;

    /** @var int */
    protected $disclaimer_id;

    /** @var Disclaimer */
    protected $disclaimer;

    /** @var string|array */
    protected $cta;

    /** @var string */
    protected $brand_id;

    /** @var int */
    protected $cid;

    /** @var int */
    protected $oid;

    /** @var string */
    protected $source;

    /** @var array */
    protected $fields;

    /** @var CacheInterface */
    protected $cache;

    /** @var MasterkeyFinder */
    protected $masterkey;

    /**  @var array|null */
    protected $dealers_list;

    /** @var Collection|null */
    protected $form_dealers;

    /** @var FormData|null */
    protected $form_data;

    /** @var array|null */
    protected $form_data_errors = [];

    /** @var array */
    protected $form_data_absent_fields = [];

    /** @var ClientParamInterface */
    protected $client_params;

    /**
     * @var bool
     */
    protected $with_dealers = false;

    /**
     * @var bool
     */
    protected $with_subdealers = true;

    /**
     * Storage for backup dealers, disclaimer etc
     * @var StorageInterface
     */
    protected $storage;

    /**
     * BaseForm constructor.
     */
    public function __construct()
    {
        $this->masterkey = new MasterkeyFinder();
    }

    /**
     * @return Disclaimer
     */
    public function getDisclaimer(): Disclaimer
    {
        return $this->disclaimer;
    }

    /**
     * @param string $form_id
     */
    public function setFormID(string $form_id)
    {
        $this->formID = $form_id;
    }

    /** @return string */
    public function getFormID(): string
    {
        return $this->formID;
    }

    /**
     * @return self
     */
    public function getForm(): self
    {
        return $this;
    }

    /**
     * @param array
     * @return array
     */
    public function getFields(): array
    {
        return $this->fields;
    }

    /**
     * @return bool
     */
    public function isEnable(): bool
    {
        return $this->enable;
    }

    /**
     * @param bool $enable
     * @return $this
     */
    public function setEnable(bool $enable): self
    {
        $this->enable = $enable;
        return $this;
    }

    /**
     * @return int
     */
    public function getDisclaimerID(): int
    {
        return $this->disclaimer_id;
    }

    /**
     * @param int $disclaimer_id
     * @return $this
     */
    public function setDisclaimerID(int $disclaimer_id): BaseForm
    {
        $this->disclaimer_id = $disclaimer_id;
        return $this;
    }

    /**
     * @return array
     */
    public function getCTA(): array
    {
        return $this->cta;
    }

    /**
     * @param array $cta
     * @return $this
     */
    public function setCTA(array $cta): BaseForm
    {
        $this->cta = $cta;
        return $this;
    }

    /**
     * @return string
     */
    public function getBrandID(): string
    {
        return $this->brand_id;
    }

    /**
     * @param string $brand_id
     * @return $this
     */
    public function setBrandID(string $brand_id): BaseForm
    {
        $this->brand_id = $brand_id;
        return $this;
    }

    /**
     * @return int
     */
    public function getCID(): int
    {
        return $this->cid;
    }

    /**
     * @param int $cid
     * @return $this
     */
    public function setCID(int $cid): self
    {
        $this->cid = $cid;
        return $this;
    }

    /**
     * @return int|null
     */
    public function getOID(): ?int
    {
        return $this->oid;
    }

    /**
     * @param int $oid
     * @return $this
     */
    public function setOID(int $oid): self
    {
        $this->oid = $oid;
        return $this;
    }

    /**
     * @return string
     */
    public function getSource(): string
    {
        return $this->source;
    }

    /**
     * @param string $source
     * @return $this
     */
    public function setSource(string $source): self
    {
        $this->source = $source;
        return $this;
    }

    /**
     * @return CacheInterface
     */
    public function getCache(): CacheInterface
    {
        return $this->cache;
    }

    /**
     * @param CacheInterface $cache
     * @return $this
     */
    public function setCache(CacheInterface $cache): self
    {
        $this->cache = $cache;
        return $this;
    }

    /**
     * @return bool
     */
    public function withDealers(): bool
    {
        return $this->with_dealers;
    }

    /**
     * @param bool $state
     * @return $this
     */
    public function setWithDealers(bool $state): self
    {
        $this->with_dealers = $state;
        return $this;
    }

    /**
     * @return array|null
     */
    public function getDealersList(): ?array
    {
        return $this->dealers_list;
    }

    /**
     * @return bool
     */
    public function withSubDealers(): bool
    {
        return $this->with_subdealers;
    }


    /**
     * @param bool $state
     * @return $this
     */
    public function setWithSubDealers(bool $state): self
    {
        $this->with_subdealers = $state;
        return $this;
    }



    public function setDealersList(array $dealers): self
    {
        $this->dealers_list = $dealers;
        return $this;
    }

    /**
     * @param array $form
     */
    abstract public function initForm(array $form);

    /**
     * @param array $form_data
     */
    public function setFormData(array $form_data)
    {
        $this->form_data = new FormData($form_data);
    }

    /**
     * @return array
     */
    public function getFormAbsentFields(): array
    {
        return $this->form_data_absent_fields;
    }

    /**
     * @return array|null
     */
    public function getFormErrors(): array
    {
        return $this->form_data_errors;
    }

    /**
     * @return bool
     */
    public function isFormDataValid(): bool
    {
        return !(count($this->getFormErrors()) or (bool) count($this->getFormAbsentFields()));
    }

    /**
     * @return ClientParamInterface
     */
    public function getClientParams(): ClientParamInterface
    {
        return $this->client_params;
    }

    /**
     * @param ClientParamInterface $client_params
     * @return $this
     */
    public function setClientParams(ClientParamInterface $client_params): self
    {
        $this->client_params = $client_params;
        return $this;
    }

    /**
     * @param StorageInterface $storage
     */
    public function setStorage(StorageInterface $storage): void
    {
        $this->storage = $storage;
    }

    /**
     * @return StorageInterface|null
     */
    public function getStorage(): ?StorageInterface
    {
        return $this->storage;
    }

    /**
     * @return MasterkeyFinder
     */
    public function getMasterkey(): MasterkeyFinder
    {
        return $this->masterkey;
    }
}
