<?php
/** @noinspection ALL */
declare(strict_types=1);

namespace FCAPoland\FormsLib\Resource;

/**
 * Class Dealer
 * @package FCAPoland\FormsLib\Resource
 */
class Dealer
{
    /**
     * List of object's properties that can be directly returned when invoking `asArray()` method
     *
     * @var array
     */
    protected $properties = array();

    /**
     * Dealer constructor.
     * @param $dealer_data
     */
    public function __construct($dealer_data)
    {
        $this->properties = $dealer_data;
        return $this;
    }

    /**
     * @param $dealer_data
     * @return Dealer
     */
    public function setProperties($dealer_data): Dealer
    {
        $this->properties = $dealer_data;
        return $this;
    }

    /**
     * Get property like 'code', 'sitecode', 'brand', 'name', 'type', 'address', 'website', 'email', 'phones', etc.
     *
     * @param $name
     *
     * @return mixed|null
     */
    private function getProperty($name)
    {
        return array_key_exists($name, $this->properties) ? $this->properties[$name] : null;
    }

    /**
     * Return dealer code in form: SINCOM.SITECODE
     *
     * @return string|null
     */
    public function getCode(): ?string
    {
        if ($sincom = $this->getSINCOM() and $sitecode = $this->getSiteCode()) {
            return $sincom . '.' . $sitecode;
        }
        return null;
    }

    /**
     * UID (Unique ID) is a composition of:
     *  - SINCOM (7 digits)
     *  - Site code (3 digits)
     *  - brand ID (2 digits)
     *  - service type ("sales" or "aftersales"
     *
     * This function returns this values glued (by default) with dot separator, for example:
     *
     *     0075450.001.00.sales
     *
     * @param string $separator
     *
     * @return string|null
     */
    public function getUID(string $separator = '.'): ?string
    {
        if ($sincom = $this->getSINCOM()
            and $sitecode = $this->getSiteCode()
            and $brand_id = $this->getBrandId()
            and $type = $this->getProperty('type')
        ) {
            return join($separator, array($sincom, $sitecode, $brand_id, $type));
        }
        return null;
    }

    /**
     * @return string|null
     */
    public function getSINCOM(): ?string
    {
        return $this->getProperty('code');
    }

    /**
     * @return string|null
     */
    public function getLegalEntity(): ?string
    {
        return $this->getProperty('legal_entity');
    }

    /**
     * @return string|null
     */
    public function getSiteCode(): ?string
    {
        return $this->getProperty('sitecode');
    }

    /**
     * @return string|null
     */
    public function getBrandId(): ?string
    {
        return $this->getProperty('brand');
    }

    /**
     * @return string|null
     */
    public function getName(): ?string
    {
        return $this->getProperty('name');
    }

    /**
     * @return string|null
     */
    public function getEmail(): ?string
    {
        return $this->getProperty('email');
    }

    /**
     * @return array
     */
    public function getAddress(): ?array
    {
        return $this->getProperty('address');
    }

    /**
     * @return string|null
     */
    public function getAddressStreet(): ?string
    {
        $address = (array) $this->getAddress();
        return$address['street'] ?? null;
    }

    /**
     * @return string|null
     */
    public function getAddressCity(): ?string
    {
        $address = (array) $this->getAddress();

        return $address['city'] ?? null;
    }

    /**
     * @return string|null
     */
    public function getAddressPostCode(): ?string
    {
        $address = (array) $this->getAddress();
        return $address['post_code'] ?? null;
    }

    /**
     * @return array|null
     */
    public function getAddressRegion(): ?array
    {
        $address = (array) $this->getAddress();
        return $address['region'] ?? null;
    }

    /**
     * @return int|null
     */
    public function getAddressRegionId(): ?int
    {
        $region = (array) $this->getAddressRegion();

        return $region['id'] ?? null;
    }

    /**
     * @return string|null
     */
    public function getAddressRegionName(): ?string
    {
        $region = (array) $this->getAddressRegion();
        return $region['name'] ?? null;
    }

    /**
     * @return array
     */
    public function getPhones(): array
    {
        return (array) $this->getProperty('phones');
    }

    /**
     * Gets single available phone number
     *
     * @return null|string
     */
    public function getPhone(): ?string
    {
        if ($phones = $this->getPhones()) {
            return $phones[0] ?? null;
        }
        return null;
    }

    /**
     * Gets single available phone number for freespee
     *
     * @return string|null
     */
    public function getPhoneFreespee(): ?string
    {
        if ($phone = $this->getPhone()) {
            $phone = str_replace(array( ' ', '-' ), array( '', '' ), $phone);
            if (strlen($phone) < 9) {
                return null;
            }
            return substr($phone, - 9);
        }
        return null;
    }

    /**
     * @return string|null
     */
    public function getFax(): ?string
    {
        return $this->getProperty('fax');
    }

    /**
     * @return bool
     */
    public function hasSalesService(): bool
    {
        $services = (array) $this->getProperty('services');
        return (bool) $services['sales'] ?? false;
    }

    /**
     * @return bool
     */
    public function isSubDealer(): bool
    {
        return (bool) $this->getProperty('subdealer');
    }

    /**
     * @return string|bool
     */
    public function getRZ(): ?string
    {
        return $this->getProperty('rz');
    }

    /**
     * @return string|null
     */
    public function getRZFirstName(): ?string
    {
        $rz = (array) $this->getRZ();

        return $rz['first_name'] ?? null;
    }

    /**
     * @return string|null
     */
    public function getRZLastName(): ?string
    {
        $rz = (array) $this->getRZ();
        return $rz['last_name'] ?? null;
    }

    /**
     * @return string|null
     */
    public function getRZEmail(): ?string
    {
        $rz = (array) $this->getRZ();
        return $rz['email'] ?? null;
    }

    /**
     * @return string|null
     */
    public function getRZPhone(): ?string
    {
        $rz = (array) $this->getRZ();
        return $rz['phone'] ?? null;
    }

    /**
     * @param null $type
     * @return mixed|null
     */
    public function getCoordinates($type = null)
    {
        if ($type === null) {
            return $this->getProperty('coordinates');
        }

        $coordinates = (array) $this->getCoordinates();

        if (stripos($type, 'long') !== false) {
            return $coordinates['longitude'] ?? null;
        }

        if (stripos($type, 'lat') !== false) {
            return $coordinates['latitude'] ?? null;
        }
        return null;
    }
}
