<?php
declare(strict_types=1);

namespace FCAPoland\FormsLib;

use Exception;
use FCAPoland\ApiPrivacyHelper\PrivacyPolicy;
use FCAPoland\ApiPrivacyHelper\StorageInterface;
use FCAPoland\FormsLib\Config\LibConfig;
use FCAPoland\FormsLib\Exception\FormNotFoundException;
use FCAPoland\FormsLib\Exception\FormsNotFoundException;
use FCAPoland\FormsLib\Exception\InvalidArgumentException;
use FCAPoland\FormsLib\Form\Form;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;

/**
 * Class FormManager
 * @package FCAPoland\FormsLib
 */
class FormManager
{
    use LoggerAwareTrait;

    /** @var StorageInterface */
    private $storage;

    /** @var LibConfig */
    private $lib_config;

    /** @var CacheInterface */
    private $cache;

    /** @var array */
    private $forms;

    /**
     * FormManager constructor.
     */
    public function __construct()
    {
        if (session_status() != PHP_SESSION_ACTIVE) {
            session_start();
        }
    }

    /**
     * @param LibConfig $lib_config
     * @throws FormsNotFoundException
     * @throws Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function setLibConfig(LibConfig $lib_config)
    {
        $this->lib_config = $lib_config;
        $this->storage = new Storage($this->lib_config->getStoragePath());
        $this->loadForms();
    }

    /**
     * @param CacheInterface $cache
     */
    public function setCache(CacheInterface $cache)
    {
        $this->cache = $cache;
    }

    /**
     * @param $id
     * @return Form
     * @throws FormNotFoundException
     */
    public function getFormByID($id): Form
    {
        if (!isset($this->forms[$id])) {
            throw new FormNotFoundException("Form {$id} not found");
        }
        return $this->forms[$id];
    }

    /**
     * @throws Exception
     * @throws FormsNotFoundException
     * @throws Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    private function loadForms()
    {
        $this->forms = [];
        if ($this->lib_config->getForms() and is_array($this->lib_config->getForms())) {
            foreach ($this->lib_config->getForms() as $form_config) {
                try {
                    $form = new Form();
                    if ($this->cache instanceof CacheInterface) {
                        $form->setCache($this->cache);
                    }
                    if ($this->logger instanceof LoggerInterface) {
                        $form->setLogger($this->logger);
                    }
                    $form->setStorage($this->getStorage());
                    $form->initForm($form_config);
                    $this->forms[$form->getFormID()] = $form;
                } catch (InvalidArgumentException $e) {
                    throw new Exception($e->getMessage());
                }
            }
        } else {
            throw new FormsNotFoundException('Forms not found. Please Check lib config');
        }
    }

    /**
     * @return string|null
     * @throws Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function getPrivacyPolicy(): ?string
    {
        try {
            $privacy_policy = new PrivacyPolicy($this->getStorage());
            if ($privacy_policy_hash = $this->lib_config->getPrivacyPolicyHash()) {
                $privacy_policy->setPrivacyPolicyHash($privacy_policy_hash);
            }
            if ($this->cache instanceof CacheInterface) {
                $privacy_policy->setCache($this->cache);
            }
            return $privacy_policy->getPrivacyPolicyContent() ?? null;
        } catch (Exception $e) {
            if ($this->logger instanceof LoggerInterface) {
                $this->logger->error('Can not init PrivacyPolicy object: ' . $e->getMessage());
            }
            throw new Exception('Can not get privacy');
        }
    }

    /**
     * @return StorageInterface
     */
    public function getStorage(): StorageInterface
    {
        return $this->storage;
    }

    /**
     * @param StorageInterface $storage
     */
    public function setStorage(StorageInterface $storage): void
    {
        $this->storage = $storage;
    }
}
