<?php


namespace Unit;


use FCAPoland\FormsLib\Field\AddressField;
use FCAPoland\FormsLib\Field\AppointmentDateField;
use FCAPoland\FormsLib\Field\BaseField;
use FCAPoland\FormsLib\Field\BrandIDField;
use FCAPoland\FormsLib\Field\CityField;
use FCAPoland\FormsLib\Field\CommunicationChannelField;
use FCAPoland\FormsLib\Field\CompanyField;
use FCAPoland\FormsLib\Field\CsrfField;
use FCAPoland\FormsLib\Field\CTAField;
use FCAPoland\FormsLib\Field\DealerSincomField;
use FCAPoland\FormsLib\Field\EmailField;
use FCAPoland\FormsLib\Field\FieldInterface;
use FCAPoland\FormsLib\Field\FirstNameField;
use FCAPoland\FormsLib\Field\LastNameField;
use FCAPoland\FormsLib\Field\PhoneField;
use FCAPoland\FormsLib\Field\RepeatEmailField;
use FCAPoland\FormsLib\Field\Rule\NotEmpty;
use FCAPoland\FormsLib\Field\ZipCodeField;
use FCAPoland\LeadsAPIHelper\Lead;
use PHPUnit\Framework\TestCase;

/**
 * Class FieldsTest
 * @package Unit
 */
class FieldsTest extends TestCase
{
    public function testBaseField()
    {
        $object = new BaseField();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(BaseField::class, $object->getFieldClassName());

        $this->assertIsArray($object->getErrorsMessages());
        $this->assertNull($object->getPlaceHolder());
        $this->assertNull($object->getFormat());
        $this->assertNull($object->getLabel());
        $this->assertNull($object->getOptionValues());

        $object->setPlaceHolder('AAA');
        $object->setFormat('BBB');
        $object->setLabel('CCC');
        $object->setOptionValues(['option']);

        $this->assertEquals('AAA', $object->getPlaceHolder());
        $this->assertEquals('BBB', $object->getFormat());
        $this->assertEquals('CCC', $object->getLabel());
        $this->assertIsArray($object->getOptionValues());

        // Domyślnie każde pole jest wymagane
        $this->assertTrue($object->isRequired());
        $object->setRequired(false);
        $this->assertFalse($object->isRequired());
        // BaseField nie ma żadnych zasad walidacyjnych dlatego powinien zwracać pozytywną walidacje
        $this->assertTrue($object->isValid('111'));

        // BaseField jest z nieokreślonym typem pola, dlatego getType() zwraca null który zaś nie jest prawidłowym
        // deklarowanym typem zwracanej wartości z getType() (string)
        $this->expectException(\TypeError::class);
        $this->assertNull($object->getType());
        $object->setType(BaseField::TYPE_TEXT);
        $this->assertEquals(BaseField::TYPE_TEXT, $object->getType());
        // Podobnie z getName()
        $this->expectException(\TypeError::class);
        $this->assertNull($object->getName());
        $object->setName('xxx');
        $this->assertEquals('xxx', $object->getName());

        $this->assertIsArray($object->getRules());
        $this->assertIsArray($object->getRuleObjects());
        $this->assertEmpty($object->getRules());
        $this->assertEmpty($object->getRuleObjects());

        $object->setRules([
            NotEmpty::class
        ]);

        $this->assertNotEmpty($object->getRules());
        $this->assertEmpty($object->getRuleObjects());
        // initRules dopiero zamienia zasady na obiekty zasad do późniejszej walidacji
        $object->initRules();
        $this->assertNotEmpty($object->getRuleObjects());

        $this->assertFalse($object->isValid(''));
        $this->assertTrue($object->isValid('AAA'));
        $this->assertNotEmpty($object->getErrorsMessages());
    }

    public function testAddressField()
    {
        $object = new AddressField();
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(AddressField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValid(null));
        $this->assertFalse($object->isValid(''));
        // Max 200 znakóœ
        $this->assertFalse($object->isValid('_12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890'));
        $this->assertTrue($object->isValid('Address'));
    }

    public function testAppointmentDateField()
    {
        $object = new AppointmentDateField();
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(AppointmentDateField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValid(null));
        $this->assertFalse($object->isValid(''));
        // Akceptowany format z ValidDate DD-MM-RRRR
        $this->assertFalse($object->isValid('2020-02-02'));
        $this->assertTrue($object->isValid('02-02-2020'));
    }

    public function testBrandIdField()
    {
        $object = new BrandIDField(BaseField::TYPE_SELECT, ['00' => 'Fiat']);
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(BrandIDField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValid(null));
        $this->assertFalse($object->isValid(''));
        $this->assertFalse($object->isValid('77'));
        $this->assertTrue($object->isValid('00'));
    }

    public function testCityField()
    {
        $object = new CityField();
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(CityField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValid(null));
        $this->assertFalse($object->isValid(''));
        // Max 100 znaków
        $this->assertFalse($object->isValid('_0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789'));
        $this->assertTrue($object->isValid('Wadowice'));
    }

    public function testCommunicationChannelField()
    {
        $object = new CommunicationChannelField();
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(CommunicationChannelField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValid(null));
        $this->assertFalse($object->isValid(''));
        $this->assertFalse($object->isValid('D'));
        // Akceptujemy tylko dwa typy kanałów komunikacji które są w klasie Lead
        $this->assertTrue($object->isValid(Lead::COMMUNICATION_CHANNEL_MAIL));
        $this->assertTrue($object->isValid(Lead::COMMUNICATION_CHANNEL_TELEPHONE));
    }

    public function testCompanyField()
    {
        $object = new CompanyField();
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(CompanyField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValid(null));
        $this->assertFalse($object->isValid(''));
        // Max 200 znakóœ
        $this->assertFalse($object->isValid('_12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890'));
        $this->assertTrue($object->isValid('Firma X'));
    }

    public function testCsrfField()
    {
        $object = new CsrfField('test');
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(CsrfField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValid(null));
        $this->assertFalse($object->isValid(''));
        $this->assertFalse($object->isValid('xxx'));
        // TODO walidacja na prawidłowy csrf - mock
        //$this->assertTrue($object->isValid($object->getValue()));
    }

    public function testCTAField()
    {
        $object = new CTAField(BaseField::TYPE_SELECT, ['TD' => 'JAZDA']);
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(CTAField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValid(null));
        $this->assertFalse($object->isValid(''));
        $this->assertFalse($object->isValid('RP'));
        $this->assertTrue($object->isValid('TD'));
    }

    public function testDealerSincomField()
    {
        $object = new DealerSincomField();
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(DealerSincomField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValid(null));
        $this->assertFalse($object->isValid(''));
        $this->assertFalse($object->isValid('xxxx'));
        $this->assertTrue($object->isValid('1111111.000'));
    }

    public function testEmailField()
    {
        $object = new EmailField();
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(EmailField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValid(null));
        $this->assertFalse($object->isValid(''));
        $this->assertFalse($object->isValid('xxxx'));
        $this->assertFalse($object->isValid('xxxx@localhost'));
        $this->assertFalse($object->isValid('0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789@wp.pl'));
        $this->assertTrue($object->isValid('wp@wp.pl'));
        $this->assertTrue($object->isValid('support@unicorn.net.pl'));
    }

    public function testFirstNameField()
    {
        $object = new FirstNameField();
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(FirstNameField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValid(null));
        $this->assertFalse($object->isValid(''));
        $this->assertFalse($object->isValid('1'));
        // Min 2 znaki
        $this->assertFalse($object->isValid('1'));
        $this->assertTrue($object->isValid('aa'));
        // Max 50 znaków
        $this->assertFalse($object->isValid('_01234567890123456789012345678901234567890123456789'));
        $this->assertTrue($object->isValid('01234567890123456789012345678901234567890123456789'));
        $this->assertTrue($object->isValid('Adam'));
    }

    public function testLastNameField()
    {
        $object = new LastNameField();
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(LastNameField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValid(null));
        $this->assertFalse($object->isValid(''));
        $this->assertFalse($object->isValid('1'));
        // Min 2 znaki
        $this->assertFalse($object->isValid('1'));
        $this->assertTrue($object->isValid('aa'));
        // Max 50 znaków
        $this->assertFalse($object->isValid('_01234567890123456789012345678901234567890123456789'));
        $this->assertTrue($object->isValid('01234567890123456789012345678901234567890123456789'));
        $this->assertTrue($object->isValid('Adamowski'));
    }

    public function testPhoneField()
    {
        $object = new PhoneField();
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(PhoneField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValid(null));
        $this->assertFalse($object->isValid(''));
        $this->assertFalse($object->isValid('xxxx'));
        $this->assertFalse($object->isValid('+44666666666'));
        $this->assertTrue($object->isValid('+48666666666'));
        $this->assertTrue($object->isValid('666666666'));
        $this->assertTrue($object->isValid('666-666-666'));
        $this->assertTrue($object->isValid('+48 666-666-666'));
    }


    public function testRepeatEmailField()
    {
        $object = new RepeatEmailField();
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(RepeatEmailField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValidRepeatEmail(null, null));
        $this->assertFalse($object->isValidRepeatEmail('', ''));
        $this->assertFalse($object->isValidRepeatEmail('xxxx', 'xxxx'));
        $this->assertFalse($object->isValidRepeatEmail('xxxx@localhost', 'xxxx@localhost'));
        $this->assertFalse($object->isValid('0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789@wp.pl'));
        $this->assertTrue($object->isValidRepeatEmail('wp@wp.pl', 'wp@wp.pl'));
        $this->assertTrue($object->isValidRepeatEmail('support@unicorn.net.pl', 'support@unicorn.net.pl'));
    }

    public function testZipCodeField()
    {
        $object = new ZipCodeField();
        $object->initRules();
        $this->assertInstanceOf(FieldInterface::class, $object);
        $this->assertEquals(ZipCodeField::class, $object->getFieldClassName());

        // Nie puste
        $this->assertFalse($object->isValid(null));
        $this->assertFalse($object->isValid(''));
        $this->assertFalse($object->isValid('1'));
        $this->assertFalse($object->isValid('83000'));
        $this->assertFalse($object->isValid('83 000'));
        $this->assertTrue($object->isValid('83-000'));
        $this->assertTrue($object->isValid('00-000'));
    }
}
