<?php
declare(strict_types=1);

namespace FCAPoland\FormsLib\Field;

use FCAPoland\FormsLib\Exception\FieldRuleNotFound;
use FCAPoland\FormsLib\Field\Rule\RuleInterface;
use ReflectionClass;
use ReflectionException;

/**
 * Class BaseField
 * @package FCAPoland\FormsLib\Field
 */
class BaseField implements FieldInterface
{
    const TYPE_HIDDEN = 'hidden';
    const TYPE_TEXT = 'text';
    const TYPE_TEXTAREA = 'textarea';
    const TYPE_EMAIL = 'email';
    const TYPE_DATE = 'date';
    const TYPE_PHONE = 'phone';
    const TYPE_SELECT = 'select';
    const TYPE_RADIO = 'radio';
    const TYPE_CHECKBOX = 'checkbox';
    const TYPE_NUMBER = 'number';
    const TYPE_RECAPTCHA = 'recaptcha';

    /**
     * Rules for validation
     * @var array
     */
    protected $rules = [];

    /**
     * Rules for validation
     * @var array
     */
    protected $rule_objects = [];

    /**
     * Type of form field
     * @var string
     */
    protected $type;

    /**
     * Name tag of field
     * @var string
     */
    protected $name;

    /**
     * Label of form field
     * @var string|null
     */
    protected $label;

    /**
     * Placeholder of form field
     * @var string|null
     */
    protected $place_holder;

    /**
     * Required of form field
     * @var bool
     */
    protected $required = true;

    /**
     * Format of form field
     * @var string|null
     */
    protected $format;

    /**
     * ?????????
     * @var mixed
     */
    protected $value;

    /**
     * Options for select,radio[],checkbox[]
     * @var array|null
     */
    protected $option_values;

    /** @var array */
    protected $error_messages = [];

    /**
     * List of available field types
     * @return string[]
     */
    public static function getFieldTypes(): array
    {
        return [
            self::TYPE_HIDDEN,
            self::TYPE_TEXT,
            self::TYPE_TEXTAREA,
            self::TYPE_EMAIL,
            self::TYPE_DATE,
            self::TYPE_PHONE,
            self::TYPE_SELECT,
            self::TYPE_RADIO,
            self::TYPE_CHECKBOX,
            self::TYPE_NUMBER,
            self::TYPE_RECAPTCHA,
        ];
    }

    public function __construct()
    {
        $this->setDefaultRules();
    }

    public function getValue()
    {
        return $this->value;
    }

    public function setValue($value): void
    {
        $this->value = $value;
    }

    protected function setDefaultRules(): void
    {
    }

    public function isValid($value): bool
    {
        // Reset tablicy z błędami, dla każdej z walidowanych wartości powinien być resetowany
        $this->error_messages = [];
        foreach ($this->getRuleObjects() as $rule) {
            if (!$rule->isValidValue($value)) {
                $this->error_messages[] = $rule->getMessage();
            }
        }
        return count($this->error_messages) == 0;
    }

    public function getRules(): array
    {
        return $this->rules;
    }

    public function getRuleObjects(): array
    {
        return $this->rule_objects;
    }

    public function setRuleObjects(array $rule_objects): BaseField
    {
        $this->rule_objects = $rule_objects;
        return $this;
    }

    public function getType(): string
    {
        return $this->type;
    }

    public function getName(): string
    {
        return $this->name;
    }

    public function getLabel(): ?string
    {
        return $this->label;
    }

    public function getPlaceHolder(): ?string
    {
        return $this->place_holder;
    }

    public function isRequired(): bool
    {
        return $this->required;
    }

    public function getFormat(): ?string
    {
        return $this->format;
    }

    public function getOptionValues(): ?array
    {
        return $this->option_values;
    }

    public function setRules(array $rules): void
    {
        $this->rules = $rules;
    }

    public function setType(string $type): void
    {
        $this->type = $type;
    }

    public function setName(string $name): void
    {
        $this->name = $name;
    }

    public function setLabel(string $label): void
    {
        $this->label = $label;
    }

    public function setPlaceHolder(?string  $place_holder): void
    {
        $this->place_holder = $place_holder;
    }

    public function setRequired(bool $required): void
    {
        $this->required = $required;
    }

    public function setFormat(?string $format): void
    {
        $this->format = $format;
    }

    public function setOptionValues(?array $options): void
    {
        $this->option_values = $options;
    }

    public function initRules(): void
    {
        // Reset obiektów zasad
        $this->rule_objects = [];
        foreach ($this->getRules() as $rule_object => $rule_params) {
            $rule_params = array_merge([$this], $rule_params);
            try {
                $objectReflection = new ReflectionClass($rule_object);
                /** @var RuleInterface $object */
                $object = $objectReflection->newInstanceArgs($rule_params);
                $this->rule_objects[$rule_object] = $object;
            } catch (ReflectionException $e) {
                throw new FieldRuleNotFound("Do not found rule " . get_class($rule_params));
            }
        }
    }

    public function getFieldClassName(): string
    {
        return get_class($this);
    }

    public function getErrorsMessages(): array
    {
        return $this->error_messages;
    }
}
