<?php
declare(strict_types=1);

namespace FCAPoland\FormsLib\Resource\Dealer;

use Exception;
use FCAPoland\ApiPrivacyHelper\StorageInterface;
use FCAPoland\FormsLib\Exception\FetchDealersFCAAPIException;
use FCAPoland\FormsLib\Exception\FormsLIBException;
use FCAPoland\FormsLib\Form\Form;
use FCAPoland\FormsLib\Resource\Dealer;
use Psr\SimpleCache\CacheInterface;
use Psr\SimpleCache\InvalidArgumentException;

/**
 * Class Collection
 * @package FCAPoland\FormsLib\Resource\Dealer
 */
class Collection
{
    /** @var string string */
    public static $user_agent = 'FCA FORMS LIB';

    const CACHE_KEY = 'formsLib.dealers.:BRAND:';

    const FCA_API_DEALER_URL = 'https://api.fcapoland.pl/dealers?types=sales&brands=:BRAND:';

    /** @var string  */
    private $brand_id;

    /** @var CacheInterface */
    private $cache;

    /** @var StorageInterface */
    private $storage;

    /** @var Form */
    private $formObject;

    /** @var array */
    private $dealers = [];

    /** @var bool  */
    private $with_subdealers = true;

    public function __construct(Form $form)
    {
        $this->brand_id = $form->getBrandID();
        $this->formObject = $form;
        if ($this->formObject->getCache() instanceof CacheInterface) {
            $this->cache = $this->formObject->getCache();
        }
        $this->setStorage($form->getStorage());
    }

    public function getDealers(): array
    {
        return $this->dealers;
    }

    public function initDealers(): self
    {
        try {
            if ($this->cache instanceof CacheInterface) {
                // z cache
                $dealers_json = $this->cache->get($this->getCacheKey(), false);
                if (!$dealers_json) {
                    // Gdy nie ma dealerów w cache pobranie z api
                    $dealers_json = $this->fetch();
                    if ($this->checkDealersJSON($dealers_json)) {
                        // Zapis nowo pobranych dealerów do cache
                        $this->cache->set($this->getCacheKey(), $dealers_json, 600);
                        // Zapis nowo pobranych dealerów do BC cache
                        $this->getStorage()->saveDealers($this->brand_id, $dealers_json);
                    }
                }
            } else {
                // pobranie dealerów z api
                $dealers_json = $this->fetch();
                // Zapis nowo pobranych dealerów do BC cache
                $this->getStorage()->saveDealers($this->brand_id, $dealers_json);
            }
        } catch (FetchDealersFCAAPIException $e) {
            $dealers_json = $this->getStorage()->getDealers($this->brand_id);
        } catch (InvalidArgumentException | Exception $e) {
            throw new FormsLIBException($e->getMessage(), 0, $e);
        }

        if (!$this->checkDealersJSON($dealers_json)) {
            throw new FormsLIBException('Dealers data is incorrect. Can not works without dealers.');
        }

        $dealers_as_array = json_decode($dealers_json, true);
        foreach ($dealers_as_array as $dealer) {
            if (!empty($this->formObject->getDisabledDealersList())) {
                if (in_array($dealer['code'], $this->formObject->getDisabledDealersList())) {
                    continue;
                }
            }
            if ($this->formObject->withDealers() and is_array($this->formObject->getDealersList())) {
                if (!in_array($dealer['code'], $this->formObject->getDealersList())) {
                    continue;
                }
            }
            $dealer_for_save = new Dealer($dealer);
            if ($this->withSubDealers() === false) {
                if ($dealer_for_save->isSubDealer()) {
                    continue;
                }
            }
            $this->dealers[$dealer['code'] . ':' . $dealer['sitecode']] = $dealer_for_save;
        }
        return $this;
    }

    private function checkDealersJSON(string $json): bool
    {
        json_decode($json);
        return (json_last_error() == JSON_ERROR_NONE);
    }

    private function getFCAAPIDealerURL(): string
    {
        return str_replace(':BRAND:', $this->brand_id, self::FCA_API_DEALER_URL);
    }

    private function getCacheKey(): string
    {
        return md5(str_replace(':BRAND:', $this->brand_id, self::CACHE_KEY));
    }

    private function getBackupCacheKey(): string
    {
        return str_replace(':BRAND:', $this->brand_id, self::CACHE_KEY);
    }

    private function fetch(): string
    {
        try {
            $curl_handle = curl_init();
            curl_setopt($curl_handle, CURLOPT_URL, $this->getFCAAPIDealerURL());
            curl_setopt($curl_handle, CURLOPT_CONNECTTIMEOUT, 15);
            curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl_handle, CURLOPT_USERAGENT, Collection::$user_agent);
            $query = curl_exec($curl_handle);
            $httpcode = curl_getinfo($curl_handle, CURLINFO_HTTP_CODE);
            curl_close($curl_handle);

            if ($query === false or $httpcode >= 400) {
                throw new FetchDealersFCAAPIException('Unable to fetch dealers data from FCA Dealer API');
            }

            if (!$this->checkDealersJSON($query)) {
                throw new FetchDealersFCAAPIException('Dealers data fetched from FCA Dealer API is incorrect!');
            }

            return $query;
        } catch (\Exception $e) {
            throw new FetchDealersFCAAPIException(
                'Unable to fetch dealers data from FCA Dealer API. ERROR:' . $e->getMessage()
            );
        }
    }

    public function withSubDealers(): bool
    {
        return $this->with_subdealers;
    }

    public function setWithSubDealers(bool $with_subdealers): void
    {
        $this->with_subdealers = $with_subdealers;
    }

    public function getStorage(): StorageInterface
    {
        return $this->storage;
    }

    public function setStorage(StorageInterface $storage): void
    {
        $this->storage = $storage;
    }
}
