<?php
declare(strict_types=1);

namespace FCAPoland\FormsLib;

use FCAPoland\FormsLib\Field\AddressField;
use FCAPoland\FormsLib\Field\AppointmentDateField;
use FCAPoland\FormsLib\Field\BrandIDField;
use FCAPoland\FormsLib\Field\CityField;
use FCAPoland\FormsLib\Field\CommunicationChannelField;
use FCAPoland\FormsLib\Field\CompanyField;
use FCAPoland\FormsLib\Field\CTAField;
use FCAPoland\FormsLib\Field\DealerSincomField;
use FCAPoland\FormsLib\Field\EmailField;
use FCAPoland\FormsLib\Field\EnergyTypeField;
use FCAPoland\FormsLib\Field\FirstNameField;
use FCAPoland\FormsLib\Field\LastNameField;
use FCAPoland\FormsLib\Field\ModelIDField;
use FCAPoland\FormsLib\Field\NotRequiredCompanyField;
use FCAPoland\FormsLib\Field\PhoneField;
use FCAPoland\FormsLib\Field\RecaptchaField;
use FCAPoland\FormsLib\Field\RepeatEmailField;
use FCAPoland\FormsLib\Field\Rule\CheckRecaptcha;
use FCAPoland\FormsLib\Field\Rule\InArray;
use FCAPoland\FormsLib\Field\Rule\IsEmail;
use FCAPoland\FormsLib\Field\Rule\IsPhone;
use FCAPoland\FormsLib\Field\Rule\MaxLength;
use FCAPoland\FormsLib\Field\Rule\MinLength;
use FCAPoland\FormsLib\Field\Rule\NotEmpty;
use FCAPoland\FormsLib\Field\Rule\RuleInterface;
use FCAPoland\FormsLib\Field\Rule\ValidDate;
use FCAPoland\FormsLib\Field\Rule\ValidDealerSincom;
use FCAPoland\FormsLib\Field\Rule\ValidPostCode;
use FCAPoland\FormsLib\Field\ZipCodeField;

/**
 * Class Message
 * @package FCAPoland\FormsLib
 */
class Message
{
    /**
     * @var  Message  Singleton instance container
     */
    protected static $_instance;

    /**
     * @var array
     */
    protected $messages = [
        AddressField::class =>
            [
                NotEmpty::class => 'Te pole jest obowiązkowe',
                MaxLength::class  => 'Podana wartość jest zbyt długa'
            ],
        AppointmentDateField::class =>
            [
                NotEmpty::class => 'Te pole jest obowiązkowe',
                ValidDate::class => 'Niepoprawny format danych',
            ],
        CityField::class =>
            [
                NotEmpty::class => 'To pole jest obowiązkowe',
                MaxLength::class => 'Podana wartość jest zbyt długa',
            ],
        CompanyField::class =>
            [
                NotEmpty::class => 'To pole jest obowiązkowe',
                MaxLength::class => 'Podana nazwa jest zbyt długa',
            ],
        NotRequiredCompanyField::class =>
            [
                NotEmpty::class => 'To pole jest obowiązkowe',
                MaxLength::class => 'Podana nazwa jest zbyt długa',
            ],
        CTAField::class =>
            [
                NotEmpty::class=> 'Wybierz typ formularza',
                InArray::class  => 'Wybrany formularz jest nieprawidłowy'
            ],
        DealerSincomField::class =>
            [
                NotEmpty::class => 'To pole jest obowiązkowe',
                ValidDealerSincom::class => 'Niepoprawny format danych',
                InArray::class  => 'Wybrany dealer nie istnieje lub został wyłączony'
            ],
        EmailField::class =>
            [
                NotEmpty::class => 'To pole jest obowiązkowe',
                IsEmail::class => 'Adres email jest nieprawidłowy',
                MaxLength::class => 'Podana wartość jest zbyt długa',
            ],
        RepeatEmailField::class =>
            [
                NotEmpty::class => 'To pole jest obowiązkowe',
                IsEmail::class => 'Adres email jest nieprawidłowy',
                MaxLength::class => 'Podana wartość jest zbyt długa',
            ],
        FirstNameField::class =>
            [
                NotEmpty::class => 'To pole jest obowiązkowe',
                MinLength::class => 'Podana wartość jest zbyt krótka',
                MaxLength::class => 'Podana wartość jest zbyt długa',
            ],
        LastNameField::class =>
            [
                NotEmpty::class => 'To pole jest obowiązkowe',
                MinLength::class => 'Podana wartość jest zbyt krótka',
                MaxLength::class => 'Podana wartość jest zbyt długa',
            ],
        ModelIDField::class =>
            [
                NotEmpty::class => 'Wybierz model',
                InArray::class  => 'Wybrany model nie istnieje'
            ],
        BrandIDField::class =>
            [
                NotEmpty::class => 'Wybierz markę',
                InArray::class  => 'Wybrana marka nie istnieje'
            ],
        PhoneField::class =>
            [
                NotEmpty::class=> 'To pole jest obowiązkowe',
                IsPhone::class => 'Numer telefonu jest nieprawidłowy',
                MinLength::class => 'Podana wartość jest zbyt krótka',
                MaxLength::class => 'Podana wartość jest zbyt długa',
            ],
        CommunicationChannelField::class =>
            [
                NotEmpty::class => 'Wybierz formę kontaktu',
                InArray::class => 'Wybrana forma kontaktu jest nieprawidłowa',
            ],
        ZipCodeField::class =>
            [
                NotEmpty::class => 'To pole jest obowiązkowe',
                ValidPostCode::class => 'Niepoprawny format danych',
            ],
        RecaptchaField::class =>
            [
                NotEmpty::class => 'Potwierdź że jesteś człowiekiem',
                CheckRecaptcha::class => 'Potwierdź że jesteś człowiekiem',
            ],
        EnergyTypeField::class =>
            [
                NotEmpty::class => 'Wybierz typ napędu',
                InArray::class => 'Wybrany typ napędu jest nieprawidłowy',
            ],
    ];

    public static function getInstance(): Message
    {
        if (Message::$_instance === null) {
            Message::$_instance = new Message();
        }

        return Message::$_instance;
    }

    public function setMessages(array $messages): Message
    {
        $this->messages = array_replace_recursive($this->messages, $messages);
        return $this;
    }

    public function getMessageFor(RuleInterface $rule): string
    {
        return isset($this->messages[$rule->getFieldClassName()][$rule->getRuleName()])
            ? $this->messages[$rule->getFieldClassName()][$rule->getRuleName()]
            : 'Not found message for ' . $rule->getFieldClassName() . ' - ' . $rule->getRuleName();
    }
}
