<?php
declare(strict_types=1);

namespace FCAPoland\FormsLib;

use Exception;
use FCAPoland\ApiPrivacyHelper\StorageInterface;

/**
 * Class Storage
 * @package FCAPoland\FormsLib
 */
class Storage implements StorageInterface
{
    const STORAGE_FILE = 'helpers_cache/privacy_helper_storage';

    private array $storage = [];

    private ?string $storage_file;

    public function __construct($storage_file = null)
    {
        $this->storage_file = $storage_file ?? self::STORAGE_FILE;
        if (!file_exists($this->storage_file)) {
            $this->createStorageFile();
        }
        $this->readStorage();
    }

    public function saveDisclaimer($id, $data): void
    {
        $this->readStorage();
        $this->storage['disclaimer'][$id] = $data;
        $this->saveStorage();
    }

    public function getDisclaimer(int $id): ?string
    {
        $this->readStorage();
        return $this->storage['disclaimer'][$id] ?? null;
    }

    public function savePrivacyPolicy(string $hash, $data): void
    {
        $this->readStorage();
        $this->storage['privacy_policy'][$hash] = $data;
        $this->saveStorage();
    }

    public function getPrivacyPolicy(string $hash): ?string
    {
        $this->readStorage();
        return $this->storage['privacy_policy'][$hash] ?? null;
    }

    private function readStorage(): void
    {
        $storage_file = file_get_contents($this->storage_file);
        if ($storage_file === false) {
            throw new Exception('Can not read storage file');
        }
        if (strlen($storage_file) == 0) {
            $this->storage = [];
        } else {
            $this->storage = json_decode($storage_file, true);
        }
    }

    public function saveToStorage(string $key, $data): void
    {
        $this->readStorage();
        $this->storage[$key] = $data;
        $this->saveStorage();
    }

    public function readFromStorage(string $key)
    {
        $this->readStorage();
        return $this->storage[$key] ?? null;
    }

    private function saveStorage(): void
    {
        if (!is_array($this->storage)) {
            $this->storage = [];
        }
        $json_storage = json_encode($this->storage);
        file_put_contents($this->storage_file, $json_storage, LOCK_EX);
    }

    public function saveDealers(string $form_and_brand_id, $data): void
    {
        $this->readStorage();
        $this->storage['dealers'][$form_and_brand_id] = $data;
        $this->saveStorage();
    }

    public function getDealers(string $form_and_brand_id): ?string
    {
        return $this->storage['dealers'][$form_and_brand_id] ?? null;
    }

    private function createStorageFile(): void
    {
        if (!file_exists($this->storage_file)) {
            $storage_file = fopen($this->storage_file, 'w');
            fclose($storage_file);
            chmod($this->storage_file, 0664);
        }
    }

    public function removeStorage(): void
    {
        if (file_exists($this->storage_file)) {
            unlink($this->storage_file);
            $this->storage = [];
        }
    }
}
