<?php
declare(strict_types=1);

namespace FCAPoland\FormsLib\Form;

use FCAPoland\ApiPrivacyHelper\Disclaimer;
use FCAPoland\ApiPrivacyHelper\StorageInterface;
use FCAPoland\FormsLib\ClientParam\ClientParamInterface;
use FCAPoland\FormsLib\Resource\Dealer\Collection;
use FCAPoland\LeadsAPIHelper\MasterkeyFinder;
use Psr\Log\LoggerAwareTrait;
use Psr\SimpleCache\CacheInterface;

/**
 * Class BaseForm
 * @package FCAPoland\FormsLib
 */
abstract class BaseForm
{
    use LoggerAwareTrait;

    /**
     * @var string
     */
    protected $formID;

    /** @var bool  */
    protected $enable = true;

    /** @var int */
    protected $disclaimer_id;

    /** @var Disclaimer */
    protected $disclaimer;

    /** @var string|array */
    protected $cta;

    /** @var string */
    protected $brand_id;

    /** @var int */
    protected $cid;

    /** @var int */
    protected $oid;

    /** @var string */
    protected $source;

    /** @var array */
    protected $fields;

    /** @var CacheInterface */
    protected $cache;

    protected \FCAPoland\LeadsAPIHelper\MasterkeyFinder $masterkey;

    /**  @var array|null */
    protected $dealers_list;

    /**  @var array|null */
    protected $disabled_dealers_list;

    /** @var Collection|null */
    protected $form_dealers;

    /** @var FormData|null */
    protected $form_data;

    /** @var array|null */
    protected $form_data_errors = [];

    /** @var array */
    protected $form_data_absent_fields = [];

    /** @var ClientParamInterface|null */
    protected $client_params;

    /**
     * @var bool
     */
    protected $with_dealers = false;

    /**
     * @var bool
     */
    protected $with_subdealers = true;

    /**
     * Storage for backup dealers, disclaimer etc
     * @var StorageInterface
     */
    protected $storage;

    /**
     * Currently this URL is only passed to privacy-api-helper (Disclaimer, PrivacyPolicy)
     */
    private string $api_base_url = 'https://api.fcapoland.pl/';

    public function __construct()
    {
        $this->masterkey = new MasterkeyFinder();
    }

    public function getDisclaimer(): Disclaimer
    {
        return $this->disclaimer;
    }

    public function setFormID(string $form_id): void
    {
        $this->formID = $form_id;
    }

    public function getFormID(): string
    {
        return $this->formID;
    }

    public function getForm(): self
    {
        return $this;
    }

    public function getFields(): array
    {
        return $this->fields;
    }

    public function isEnable(): bool
    {
        return $this->enable;
    }

    public function setEnable(bool $enable): self
    {
        $this->enable = $enable;
        return $this;
    }

    public function getDisclaimerID(): int
    {
        return $this->disclaimer_id;
    }

    public function setDisclaimerID(int $disclaimer_id): BaseForm
    {
        $this->disclaimer_id = $disclaimer_id;
        return $this;
    }

    public function getCTA(): array
    {
        return $this->cta;
    }

    public function setCTA(array $cta): BaseForm
    {
        $this->cta = $cta;
        return $this;
    }

    public function getBrandID(): string
    {
        return $this->brand_id;
    }

    public function setBrandID(string $brand_id): BaseForm
    {
        $this->brand_id = $brand_id;
        return $this;
    }

    public function getCID(): int
    {
        return $this->cid;
    }

    public function setCID(int $cid): self
    {
        $this->cid = $cid;
        return $this;
    }

    public function getOID(): ?int
    {
        return $this->oid;
    }

    public function setOID(int $oid): self
    {
        $this->oid = $oid;
        return $this;
    }

    public function getSource(): string
    {
        return $this->source;
    }

    public function setSource(string $source): self
    {
        $this->source = $source;
        return $this;
    }

    public function getCache(): CacheInterface
    {
        return $this->cache;
    }

    public function setCache(CacheInterface $cache): self
    {
        $this->cache = $cache;
        return $this;
    }

    public function withDealers(): bool
    {
        return $this->with_dealers;
    }

    public function setWithDealers(bool $state): self
    {
        $this->with_dealers = $state;
        return $this;
    }

    public function getDealersList(): ?array
    {
        return $this->dealers_list;
    }

    public function getDisabledDealersList(): ?array
    {
        return $this->disabled_dealers_list;
    }

    public function withSubDealers(): bool
    {
        return $this->with_subdealers;
    }

    public function setWithSubDealers(bool $state): self
    {
        $this->with_subdealers = $state;
        return $this;
    }

    public function setDealersList(array $dealers): self
    {
        $this->dealers_list = $dealers;
        return $this;
    }

    public function setDisabledDealersList(array $disabled_dealers_list): self
    {
        $this->disabled_dealers_list = $disabled_dealers_list;
        return $this;
    }

    abstract public function initForm(array $form);

    public function setFormData(array $form_data): void
    {
        $this->form_data = new FormData($form_data);
    }

    public function getFormAbsentFields(): array
    {
        return $this->form_data_absent_fields;
    }

    public function getFormErrors(): array
    {
        return $this->form_data_errors;
    }

    public function isFormDataValid(): bool
    {
        return !(count($this->getFormErrors()) or (bool) count($this->getFormAbsentFields()));
    }

    public function getClientParams(): ?ClientParamInterface
    {
        return $this->client_params;
    }

    public function setClientParams(ClientParamInterface $clientParam): self
    {
        $this->client_params = $clientParam;
        return $this;
    }

    public function setStorage(StorageInterface $storage): void
    {
        $this->storage = $storage;
    }

    public function getStorage(): ?StorageInterface
    {
        return $this->storage;
    }

    public function getMasterkey(): MasterkeyFinder
    {
        return $this->masterkey;
    }

    /**
     * Set FCA API base URL different from default (which is: `https://api.fcapoland.pl/`)
     */
    public function setApiBaseUrl(string $apiBaseUrl): void
    {
        $this->api_base_url = rtrim($apiBaseUrl, '/') . '/';
    }

    public function getApiBaseUrl(): string
    {
        return $this->api_base_url;
    }
}
