<?php
/** @noinspection ALL */
declare(strict_types=1);

namespace FCAPoland\FormsLib\Resource;

/**
 * Class Dealer
 * @package FCAPoland\FormsLib\Resource
 */
class Dealer
{
    /**
     * @param array $properties List of object's properties that can be directly returned when invoking `asArray()` method
     */
    public function __construct(protected array $properties)
    {
        return $this;
    }

    public function setProperties(array $dealer_data): Dealer
    {
        $this->properties = $dealer_data;
        return $this;
    }

    /**
     * Get property like 'code', 'sitecode', 'brand', 'name', 'type', 'address', 'website', 'email', 'phones', etc.
     */
    private function getProperty(string $name)
    {
        return $this->properties[$name] ?? null;
    }

    public function getCode(): ?string
    {
        if ($sincom = $this->getSINCOM() and $sitecode = $this->getSiteCode()) {
            return $sincom . '.' . $sitecode;
        }
        return null;
    }

    /**
     * UID (Unique ID) is a composition of:
     *  - SINCOM (7 digits)
     *  - Site code (3 digits)
     *  - brand ID (2 digits)
     *  - service type ("sales" or "aftersales"
     *
     * This function returns this values glued (by default) with dot separator, for example:
     *
     *     0075450.001.00.sales
     *
     */
    public function getUID(string $separator = '.'): ?string
    {
        if ($sincom = $this->getSINCOM()
            and $sitecode = $this->getSiteCode()
            and $brand_id = $this->getBrandId()
            and $type = $this->getProperty('type')
        ) {
            return implode($separator, [$sincom, $sitecode, $brand_id, $type]);
        }
        return null;
    }

    public function getSINCOM(): ?string
    {
        return $this->getProperty('code');
    }

    public function getLegalEntity(): ?string
    {
        return $this->getProperty('legal_entity');
    }

    public function getSiteCode(): ?string
    {
        return $this->getProperty('sitecode');
    }

    public function getBrandId(): ?string
    {
        return $this->getProperty('brand');
    }

    public function getName(): ?string
    {
        return $this->getProperty('name');
    }

    public function getEmail(): ?string
    {
        return $this->getProperty('email');
    }

    public function getAddress(): ?array
    {
        return $this->getProperty('address');
    }

    public function getAddressStreet(): ?string
    {
        $address = (array) $this->getAddress();
        return$address['street'] ?? null;
    }

    public function getAddressCity(): ?string
    {
        $address = (array) $this->getAddress();
        return $address['city'] ?? null;
    }

    public function getAddressPostCode(): ?string
    {
        $address = (array) $this->getAddress();
        return $address['post_code'] ?? null;
    }

    public function getAddressRegion(): ?array
    {
        $address = (array) $this->getAddress();
        return $address['region'] ?? null;
    }

    public function getAddressRegionId(): ?int
    {
        $region = (array) $this->getAddressRegion();

        return $region['id'] ?? null;
    }

    public function getAddressRegionName(): ?string
    {
        $region = (array) $this->getAddressRegion();
        return $region['name'] ?? null;
    }

    public function getPhones(): array
    {
        return (array) $this->getProperty('phones');
    }

    public function getPhone(): ?string
    {
        if (($phones = $this->getPhones()) !== []) {
            return $phones[0] ?? null;
        }
        return null;
    }

    public function getPhoneFreespee(): ?string
    {
        if (($phone = $this->getPhone()) !== null && ($phone = $this->getPhone()) !== '' && ($phone = $this->getPhone()) !== '0') {
            $phone = str_replace([' ', '-'], ['', ''], $phone);
            if (strlen($phone) < 9) {
                return null;
            }
            return substr($phone, - 9);
        }
        return null;
    }

    public function getFax(): ?string
    {
        return $this->getProperty('fax');
    }

    public function hasSalesService(): bool
    {
        $services = (array) $this->getProperty('services');
        return (bool) $services['sales'] ?? false;
    }

    public function isSubDealer(): bool
    {
        return (bool) $this->getProperty('subdealer');
    }

    public function getRZ(): ?string
    {
        return $this->getProperty('rz');
    }

    public function getRZFirstName(): ?string
    {
        $rz = (array) $this->getRZ();

        return $rz['first_name'] ?? null;
    }

    public function getRZLastName(): ?string
    {
        $rz = (array) $this->getRZ();
        return $rz['last_name'] ?? null;
    }

    public function getRZEmail(): ?string
    {
        $rz = (array) $this->getRZ();
        return $rz['email'] ?? null;
    }

    public function getRZPhone(): ?string
    {
        $rz = (array) $this->getRZ();
        return $rz['phone'] ?? null;
    }

    public function getCoordinates($type = null)
    {
        if ($type === null) {
            return $this->getProperty('coordinates');
        }
        $coordinates = (array) $this->getCoordinates();
        if (stripos((string) $type, 'long') !== false) {
            return $coordinates['longitude'] ?? null;
        }
        if (stripos((string) $type, 'lat') !== false) {
            return $coordinates['latitude'] ?? null;
        }
        return null;
    }
}
