<?php


namespace Unit;


use FCAPoland\FormsLib\Field\AddressField;
use FCAPoland\FormsLib\Field\AppointmentDateField;
use FCAPoland\FormsLib\Field\BaseField;
use FCAPoland\FormsLib\Field\BrandIDField;
use FCAPoland\FormsLib\Field\CityField;
use FCAPoland\FormsLib\Field\CommunicationChannelField;
use FCAPoland\FormsLib\Field\CompanyField;
use FCAPoland\FormsLib\Field\CsrfField;
use FCAPoland\FormsLib\Field\CTAField;
use FCAPoland\FormsLib\Field\DealerSincomField;
use FCAPoland\FormsLib\Field\EmailField;
use FCAPoland\FormsLib\Field\FieldInterface;
use FCAPoland\FormsLib\Field\FirstNameField;
use FCAPoland\FormsLib\Field\LastNameField;
use FCAPoland\FormsLib\Field\PhoneField;
use FCAPoland\FormsLib\Field\RepeatEmailField;
use FCAPoland\FormsLib\Field\Rule\NotEmpty;
use FCAPoland\FormsLib\Field\ZipCodeField;
use FCAPoland\LeadsAPIHelper\Lead;
use PHPUnit\Framework\TestCase;

/**
 * Class FieldsTest
 * @package Unit
 */
class FieldsTest extends TestCase
{
    public function testBaseField(): void
    {
        $baseField = new BaseField();
        $this->assertInstanceOf(FieldInterface::class, $baseField);
        $this->assertEquals(BaseField::class, $baseField->getFieldClassName());

        $this->assertIsArray($baseField->getErrorsMessages());
        $this->assertNull($baseField->getPlaceHolder());
        $this->assertNull($baseField->getFormat());
        $this->assertNull($baseField->getLabel());
        $this->assertNull($baseField->getOptionValues());

        $baseField->setPlaceHolder('AAA');
        $baseField->setFormat('BBB');
        $baseField->setLabel('CCC');
        $baseField->setOptionValues(['option']);

        $this->assertEquals('AAA', $baseField->getPlaceHolder());
        $this->assertEquals('BBB', $baseField->getFormat());
        $this->assertEquals('CCC', $baseField->getLabel());
        $this->assertIsArray($baseField->getOptionValues());

        // Domyślnie każde pole jest wymagane
        $this->assertTrue($baseField->isRequired());
        $baseField->setRequired(false);
        $this->assertFalse($baseField->isRequired());
        // BaseField nie ma żadnych zasad walidacyjnych dlatego powinien zwracać pozytywną walidacje
        $this->assertTrue($baseField->isValid('111'));

        // BaseField jest z nieokreślonym typem pola, dlatego getType() zwraca null który zaś nie jest prawidłowym
        // deklarowanym typem zwracanej wartości z getType() (string)
        $this->expectException(\TypeError::class);
        $this->assertNull($baseField->getType());
        $baseField->setType(BaseField::TYPE_TEXT);
        $this->assertEquals(BaseField::TYPE_TEXT, $baseField->getType());
        // Podobnie z getName()
        $this->expectException(\TypeError::class);
        $this->assertNull($baseField->getName());
        $baseField->setName('xxx');
        $this->assertEquals('xxx', $baseField->getName());

        $this->assertIsArray($baseField->getRules());
        $this->assertIsArray($baseField->getRuleObjects());
        $this->assertEmpty($baseField->getRules());
        $this->assertEmpty($baseField->getRuleObjects());

        $baseField->setRules([
            NotEmpty::class
        ]);

        $this->assertNotEmpty($baseField->getRules());
        $this->assertEmpty($baseField->getRuleObjects());
        // initRules dopiero zamienia zasady na obiekty zasad do późniejszej walidacji
        $baseField->initRules();
        $this->assertNotEmpty($baseField->getRuleObjects());

        $this->assertFalse($baseField->isValid(''));
        $this->assertTrue($baseField->isValid('AAA'));
        $this->assertNotEmpty($baseField->getErrorsMessages());
    }

    public function testAddressField(): void
    {
        $addressField = new AddressField();
        $addressField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $addressField);
        $this->assertEquals(AddressField::class, $addressField->getFieldClassName());

        // Nie puste
        $this->assertFalse($addressField->isValid(null));
        $this->assertFalse($addressField->isValid(''));
        // Max 200 znakóœ
        $this->assertFalse($addressField->isValid('_12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890'));
        $this->assertTrue($addressField->isValid('Address'));
    }

    public function testAppointmentDateField(): void
    {
        $appointmentDateField = new AppointmentDateField();
        $appointmentDateField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $appointmentDateField);
        $this->assertEquals(AppointmentDateField::class, $appointmentDateField->getFieldClassName());

        // Nie puste
        $this->assertFalse($appointmentDateField->isValid(null));
        $this->assertFalse($appointmentDateField->isValid(''));
        // Akceptowany format z ValidDate DD-MM-RRRR
        $this->assertFalse($appointmentDateField->isValid('2020-02-02'));
        $this->assertTrue($appointmentDateField->isValid('02-02-2020'));
    }

    public function testBrandIdField(): void
    {
        $brandIDField = new BrandIDField(BaseField::TYPE_SELECT, ['00' => 'Fiat']);
        $brandIDField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $brandIDField);
        $this->assertEquals(BrandIDField::class, $brandIDField->getFieldClassName());

        // Nie puste
        $this->assertFalse($brandIDField->isValid(null));
        $this->assertFalse($brandIDField->isValid(''));
        $this->assertFalse($brandIDField->isValid('77'));
        $this->assertTrue($brandIDField->isValid('00'));
    }

    public function testCityField(): void
    {
        $cityField = new CityField();
        $cityField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $cityField);
        $this->assertEquals(CityField::class, $cityField->getFieldClassName());

        // Nie puste
        $this->assertFalse($cityField->isValid(null));
        $this->assertFalse($cityField->isValid(''));
        // Max 100 znaków
        $this->assertFalse($cityField->isValid('_0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789'));
        $this->assertTrue($cityField->isValid('Wadowice'));
    }

    public function testCommunicationChannelField(): void
    {
        $communicationChannelField = new CommunicationChannelField();
        $communicationChannelField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $communicationChannelField);
        $this->assertEquals(CommunicationChannelField::class, $communicationChannelField->getFieldClassName());

        // Nie puste
        $this->assertFalse($communicationChannelField->isValid(null));
        $this->assertFalse($communicationChannelField->isValid(''));
        $this->assertFalse($communicationChannelField->isValid('D'));
        // Akceptujemy tylko dwa typy kanałów komunikacji które są w klasie Lead
        $this->assertTrue($communicationChannelField->isValid(Lead::COMMUNICATION_CHANNEL_MAIL));
        $this->assertTrue($communicationChannelField->isValid(Lead::COMMUNICATION_CHANNEL_TELEPHONE));
    }

    public function testCompanyField(): void
    {
        $companyField = new CompanyField();
        $companyField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $companyField);
        $this->assertEquals(CompanyField::class, $companyField->getFieldClassName());

        // Nie puste
        $this->assertFalse($companyField->isValid(null));
        $this->assertFalse($companyField->isValid(''));
        // Max 200 znakóœ
        $this->assertFalse($companyField->isValid('_12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890'));
        $this->assertTrue($companyField->isValid('Firma X'));
    }

    public function testCsrfField(): void
    {
        $csrfField = new CsrfField('test');
        $csrfField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $csrfField);
        $this->assertEquals(CsrfField::class, $csrfField->getFieldClassName());

        // Nie puste
        $this->assertFalse($csrfField->isValid(null));
        $this->assertFalse($csrfField->isValid(''));
        $this->assertFalse($csrfField->isValid('xxx'));
        // TODO walidacja na prawidłowy csrf - mock
        //$this->assertTrue($object->isValid($object->getValue()));
    }

    public function testCTAField(): void
    {
        $ctaField = new CTAField(BaseField::TYPE_SELECT, ['TD' => 'JAZDA']);
        $ctaField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $ctaField);
        $this->assertEquals(CTAField::class, $ctaField->getFieldClassName());

        // Nie puste
        $this->assertFalse($ctaField->isValid(null));
        $this->assertFalse($ctaField->isValid(''));
        $this->assertFalse($ctaField->isValid('RP'));
        $this->assertTrue($ctaField->isValid('TD'));
    }

    public function testDealerSincomField(): void
    {
        $dealerSincomField = new DealerSincomField();
        $dealerSincomField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $dealerSincomField);
        $this->assertEquals(DealerSincomField::class, $dealerSincomField->getFieldClassName());

        // Nie puste
        $this->assertFalse($dealerSincomField->isValid(null));
        $this->assertFalse($dealerSincomField->isValid(''));
        $this->assertFalse($dealerSincomField->isValid('xxxx'));

        $this->assertTrue($dealerSincomField->isValid('1111111.000'));
    }

    public function testDealerSincomField2(): void
    {
        $dealerSincomField = new DealerSincomField(['1111111.000' => 'TEST DEALER', '6666666.666' => '666']);
        $dealerSincomField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $dealerSincomField);
        $this->assertEquals(DealerSincomField::class, $dealerSincomField->getFieldClassName());

        $this->assertTrue($dealerSincomField->isValid('1111111.000'));
        $this->assertFalse($dealerSincomField->isValid('0000000.000'));
    }

    public function testEmailField(): void
    {
        $emailField = new EmailField();
        $emailField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $emailField);
        $this->assertEquals(EmailField::class, $emailField->getFieldClassName());

        // Nie puste
        $this->assertFalse($emailField->isValid(null));
        $this->assertFalse($emailField->isValid(''));
        $this->assertFalse($emailField->isValid('xxxx'));
        $this->assertFalse($emailField->isValid('xxxx@localhost'));
        $this->assertFalse($emailField->isValid('0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789@wp.pl'));
        $this->assertTrue($emailField->isValid('wp@wp.pl'));
        $this->assertTrue($emailField->isValid('support@unicorn.net.pl'));
    }

    public function testFirstNameField(): void
    {
        $firstNameField = new FirstNameField();
        $firstNameField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $firstNameField);
        $this->assertEquals(FirstNameField::class, $firstNameField->getFieldClassName());

        // Nie puste
        $this->assertFalse($firstNameField->isValid(null));
        $this->assertFalse($firstNameField->isValid(''));
        $this->assertFalse($firstNameField->isValid('1'));
        // Min 2 znaki
        $this->assertFalse($firstNameField->isValid('1'));
        $this->assertTrue($firstNameField->isValid('aa'));
        // Max 50 znaków
        $this->assertFalse($firstNameField->isValid('_01234567890123456789012345678901234567890123456789'));
        $this->assertTrue($firstNameField->isValid('01234567890123456789012345678901234567890123456789'));
        $this->assertTrue($firstNameField->isValid('Adam'));
    }

    public function testLastNameField(): void
    {
        $lastNameField = new LastNameField();
        $lastNameField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $lastNameField);
        $this->assertEquals(LastNameField::class, $lastNameField->getFieldClassName());

        // Nie puste
        $this->assertFalse($lastNameField->isValid(null));
        $this->assertFalse($lastNameField->isValid(''));
        $this->assertFalse($lastNameField->isValid('1'));
        // Min 2 znaki
        $this->assertFalse($lastNameField->isValid('1'));
        $this->assertTrue($lastNameField->isValid('aa'));
        // Max 50 znaków
        $this->assertFalse($lastNameField->isValid('_01234567890123456789012345678901234567890123456789'));
        $this->assertTrue($lastNameField->isValid('01234567890123456789012345678901234567890123456789'));
        $this->assertTrue($lastNameField->isValid('Adamowski'));
    }

    public function testPhoneField(): void
    {
        $phoneField = new PhoneField();
        $phoneField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $phoneField);
        $this->assertEquals(PhoneField::class, $phoneField->getFieldClassName());

        // Nie puste
        $this->assertFalse($phoneField->isValid(null));
        $this->assertFalse($phoneField->isValid(''));
        $this->assertFalse($phoneField->isValid('xxxx'));
        $this->assertFalse($phoneField->isValid('+44666666666'));
        $this->assertTrue($phoneField->isValid('+48666666666'));
        $this->assertTrue($phoneField->isValid('666666666'));
        $this->assertTrue($phoneField->isValid('666-666-666'));
        $this->assertTrue($phoneField->isValid('+48 666-666-666'));
    }


    public function testRepeatEmailField(): void
    {
        $repeatEmailField = new RepeatEmailField();
        $repeatEmailField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $repeatEmailField);
        $this->assertEquals(RepeatEmailField::class, $repeatEmailField->getFieldClassName());

        // Nie puste
        $this->assertFalse($repeatEmailField->isValidRepeatEmail(null, null));
        $this->assertFalse($repeatEmailField->isValidRepeatEmail('', ''));
        $this->assertFalse($repeatEmailField->isValidRepeatEmail('xxxx', 'xxxx'));
        $this->assertFalse($repeatEmailField->isValidRepeatEmail('xxxx@localhost', 'xxxx@localhost'));
        $this->assertFalse($repeatEmailField->isValid('0123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789@wp.pl'));
        $this->assertTrue($repeatEmailField->isValidRepeatEmail('wp@wp.pl', 'wp@wp.pl'));
        $this->assertTrue($repeatEmailField->isValidRepeatEmail('support@unicorn.net.pl', 'support@unicorn.net.pl'));
    }

    public function testZipCodeField(): void
    {
        $zipCodeField = new ZipCodeField();
        $zipCodeField->initRules();
        $this->assertInstanceOf(FieldInterface::class, $zipCodeField);
        $this->assertEquals(ZipCodeField::class, $zipCodeField->getFieldClassName());

        // Nie puste
        $this->assertFalse($zipCodeField->isValid(null));
        $this->assertFalse($zipCodeField->isValid(''));
        $this->assertFalse($zipCodeField->isValid('1'));
        $this->assertFalse($zipCodeField->isValid('83000'));
        $this->assertFalse($zipCodeField->isValid('83 000'));
        $this->assertTrue($zipCodeField->isValid('83-000'));
        $this->assertTrue($zipCodeField->isValid('00-000'));
    }
}
