<?php
declare(strict_types=1);

namespace FCAPoland\FormsLib\Form;

use FCAPoland\ApiPrivacyHelper\Disclaimer;
use FCAPoland\ApiPrivacyHelper\StorageInterface;
use FCAPoland\FormsLib\ClientParam\ClientParamInterface;
use FCAPoland\FormsLib\ClientParam\SessionClientParam;
use FCAPoland\FormsLib\Config\FCAAPIAuth;
use FCAPoland\FormsLib\Exception\FieldInitException;
use FCAPoland\FormsLib\Exception\FormsLIBException;
use FCAPoland\FormsLib\Exception\InvalidArgumentException;
use FCAPoland\FormsLib\Field\CsrfField;
use FCAPoland\FormsLib\Field\DealerSincomField;
use FCAPoland\FormsLib\Field\EmailField;
use FCAPoland\FormsLib\Field\FieldInterface;
use FCAPoland\FormsLib\Field\PhoneField;
use FCAPoland\FormsLib\Field\RecaptchaField;
use FCAPoland\FormsLib\Field\RepeatEmailField;
use FCAPoland\FormsLib\Field\Rule\RuleInterface;
use FCAPoland\FormsLib\Resource\Dealer;
use FCAPoland\FormsLib\Resource\Dealer\Collection;
use FCAPoland\LeadsAPIHelper\Lead;
use FCAPoland\LeadsAPIHelper\Request\Create;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;
use ReflectionClass;
use ReflectionException;

/**
 * Class Form
 * @package FCAPoland\FormsLib
 */
class Form extends BaseForm
{
    /**
     * @var bool
     */
    protected $skipCallCenterVerification = false;

    public function initForm(array $form): void
    {
        if (!isset($form['id']) or
            !isset($form['enabled']) or
            !isset($form['with_subdealers']) or
            !isset($form['disclaimer_id']) or
            !isset($form['cta']) or
            !isset($form['brand_id']) or
            !isset($form['cid']) or
            !isset($form['source']) or
            !isset($form['fields'])
        ) {
            throw new InvalidArgumentException('Incorrect form config, missing data');
        }

        if (!is_array($form['cta'])) {
            throw new InvalidArgumentException('Incorrect form config - CTA must be array');
        }

        if ($this->cache instanceof CacheInterface) {
            $this->masterkey->setCache($this->cache);
        }

        if ($this->logger instanceof LoggerInterface) {
            $this->masterkey->setLogger($this->logger);
        }

        $this->setFormID($form['id']);
        $this->setEnable((bool) $form['enabled']);
        $this->setDisclaimerID($form['disclaimer_id']);
        $this->setCTA($form['cta']);
        $this->setBrandID($form['brand_id']);
        $this->setCID($form['cid']);
        $this->setSource($form['source']);

        if (isset($form['oid']) and is_int($form['oid'])) {
            $this->setOID($form['oid']);
        }

        if (isset($form['with_dealers'])
            and $form['with_dealers'] == true
        ) {
            $this->setWithDealers(true);
        }
        if (isset($form['with_subdealers'])
            and $form['with_subdealers'] == false
        ) {
            $this->setWithSubDealers(false);
        }
        if (isset($form['dealers_list'])
             and is_array($form['dealers_list'])
        ) {
            $this->setDealersList($form['dealers_list']);
        }
        if (isset($form['disabled_dealers_list'])
            and is_array($form['disabled_dealers_list'])
        ) {
            $this->setDisabledDealersList($form['disabled_dealers_list']);
        }

        $this->masterkey->setBrand($this->getBrandID());
        $this->masterkey->setAppCid($this->getCID());
        if ($this->getOID() !== null && $this->getOID() !== 0) {
            $this->masterkey->setAppOid($this->getOID());
        }
        $this->masterkey->setAppSource($this->getSource());

        if (!is_array($form['fields'])) {
            throw new InvalidArgumentException('Incorrect field config for ' . $this->getFormID());
        }
        $this->initDealers();
        $this->initFieldsObjects($form['fields']);
        $this->initDisclaimer();
    }

    public function getFormArray(): array
    {
        return [
            'id' => $this->getFormID(),
            'cta' => $this->getCTA(),
            'disclaimer_id' => $this->getDisclaimer()->getDisclaimerID(),
            'fields' => $this->getFormFieldsAsArray(true),
            'agreements' => $this->getDisclaimer()->getAgreements()->getFormElements(),
            'required_fields' => $this->getRequiredFields(),
            'disclaimer' => [
                'information'   => [
                    'keys' => $this->getDisclaimer()->getInformation()->getInformationKeys(),
                    'content' => $this->getDisclaimer()->getInformation()->getInformations(),
                ],
            ],
            'form_dealers' => $this->getDealersFormArray()
        ];
    }

    private function getFormFieldsAsArray(bool $with_keys = false): array
    {
        /** @var FieldInterface $field */
        $fields = [];
        foreach ($this->fields as $field) {
            $rules = [];
            $fieldStorage = [];
            /**
             * @var  $rule
             * @var  $ruleObject RuleInterface
             */
            foreach ($field->getRuleObjects() as $ruleObject) {
                $rules[] = [
                    'rule' => $ruleObject->getRuleName(),
                    'regex' => $ruleObject->getRegex(),
                    'message' => $ruleObject->getMessage()
                ];
            }

            if (is_array($field->getOptionValues())) {
                $optionValues = [];
                foreach ($field->getOptionValues() as $k_optionValue => $optionValue) {
                    $optionValues[] = [
                        'value' => $k_optionValue,
                        'label' => $optionValue
                    ];
                }
                $fieldStorage['input'] = $optionValues;
            }

            $fieldStorage['name'] = $field->getName();
            $fieldStorage['value'] = $field->getValue();
            $fieldStorage['type'] = $field->getType();
            $fieldStorage['label'] = $field->getLabel();
            $fieldStorage['format'] = $field->getFormat();
            $fieldStorage['placeholder'] = $field->getPlaceHolder();
            $fieldStorage['required'] = $field->isRequired();
            $fieldStorage['rules'] = $rules;

            if ($field->getFieldClassName() === RecaptchaField::class) {
                /** @var $field RecaptchaField */
                $fieldStorage['recaptcha'] = [
                    'site_key' => $field->getSiteKey(),
                ];
            }

            if ($with_keys) {
                $fields[$field->getName()] = $fieldStorage;
            } else {
                $fields[] = $fieldStorage;
            }
        }
        return $fields;
    }

    public function getRequiredFields(): array
    {
        $required_fields = [];

        /** @var FieldInterface $field */
        foreach ($this->getFields() as $field) {
            if ($field->isRequired()) {
                $required_fields[] = $field->getName();
            }
        }

        return array_merge(
            $required_fields, // Wymagane pola form
            $this->disclaimer->getAgreements()->requiredFields() // wymagane pola zgód
        );
    }

    private function initFieldsObjects(array $fields): void
    {
        foreach ($fields as $field_object => $field_params) {
            try {
                if ($field_object === DealerSincomField::class) {
                    $objectReflection = new ReflectionClass($field_object);
                    /** @var FieldInterface $object */
                    $object = $objectReflection->newInstanceArgs(
                        // Add test dealer for validation
                        [$this->getDealersFormArray() + ['1111111.000' => 'TEST']]
                    );
                    $object->initRules();
                    $this->fields[$object->getName()] = $object;
                } else {
                    $objectReflection = new ReflectionClass($field_object);
                    /** @var FieldInterface $object */
                    $object = $objectReflection->newInstanceArgs($field_params);
                    $object->initRules();
                    $this->fields[$object->getName()] = $object;
                }
            } catch (ReflectionException) {
                throw new FieldInitException("Can not field init " . $field_object);
            }
        }
    }

    protected function initDisclaimer()
    {
        $this->disclaimer = new Disclaimer($this->getDisclaimerID(), $this->getStorage());
        $this->disclaimer->setApiBaseUrl($this->getApiBaseUrl());
        if ($this->getStorage() instanceof StorageInterface) {
            $this->disclaimer->setStorage($this->getStorage());
        }
        if ($this->cache instanceof CacheInterface) {
            $this->disclaimer->setCache($this->cache);
        }
        $this->disclaimer->getDisclaimer();
    }

    private function initDealers(): Form
    {
        if ($this->withDealers()) {
            $collection = new Collection($this);
            $collection->setWithSubDealers($this->withSubDealers());
            $collection->initDealers();
            $this->form_dealers = $collection;
        }
        return $this;
    }

    public function getDealersFormArray(): array
    {
        $dealers = [];
        if ($this->withDealers()) {
            /** @var Dealer $dealer */
            foreach ($this->form_dealers->getDealers() as $dealer) {
                $dealers[$dealer->getCode()] = [
                    'sincom' => $dealer->getCode(),
                    'name' => $dealer->getName(),
                    'address' => $dealer->getAddress(),
                    'coordinates' => $dealer->getCoordinates(),
                    'phones' => $dealer->getPhones(),
                    'email' => $dealer->getEmail()
                ];
            }
            return $dealers;
        }
        return [];
    }

    private function checkFormData(): void
    {
        /** @var FieldInterface $field */
        foreach ($this->fields as $field) {
            if ($field instanceof RepeatEmailField) {
                if (!$field->isValid($this->form_data->getPropertyByField($field)) or
                    !$field->isValidRepeatEmail(
                        $this->form_data->getPropertyByField($field),
                        $this->form_data->getPropertyByField(new EmailField())
                    )
                ) {
                    $this->form_data_errors[$field->getName()] = $field->getErrorsMessages();
                }
            } elseif ($field instanceof PhoneField) {
                if (isset($this->fields['communication_channel']) and
                    $this->form_data->getPropertyByField($this->fields['communication_channel']) == Lead::COMMUNICATION_CHANNEL_MAIL) {
                    if (!empty($this->form_data->getPropertyByField($this->fields['phone'])) && !$field->isValid($this->form_data->getPropertyByField($field))) {
                        $this->form_data_errors[$field->getName()] = $field->getErrorsMessages();
                    }
                } elseif (!$field->isValid($this->form_data->getPropertyByField($field))) {
                    $this->form_data_errors[$field->getName()] = $field->getErrorsMessages();
                }
            } elseif (!$field->isValid($this->form_data->getPropertyByField($field))) {
                $this->form_data_errors[$field->getName()] = $field->getErrorsMessages();
            }
        }
    }

    public function setFormData(array $form_data): void
    {
        parent::setFormData($form_data);
        $this->checkRequiredFields();
        $this->checkFormData();
    }

    private function checkRequiredFields(): void
    {
        foreach ($this->getRequiredFields() as $requiredField) {
            if (isset($this->fields['communication_channel']) and
                $this->form_data->getPropertyByField($this->fields['communication_channel']) == Lead::COMMUNICATION_CHANNEL_MAIL
                and $requiredField === 'phone'
            ) {
                continue;
            }
            if (!$this->form_data->hasProperty($requiredField)) {
                $this->form_data_absent_fields[] = $requiredField;
            }
        }
    }

    public function submitForm(FCAAPIAuth $fcaapiAuth, bool $debug_saved_data = false): array
    {
        if (!$this->isEnable()) {
            throw new FormsLIBException('Can not submit disabled form');
        }

        if (!$this->isFormDataValid()) {
            throw new FormsLIBException('Can not submit not valid form');
        }

        if (!($this->client_params instanceof ClientParamInterface)) {
            $this->client_params = new SessionClientParam();
        }

        if ($this->client_params->getCID() !== null && $this->client_params->getCID() !== 0) {
            $this->masterkey->setCid($this->client_params->getCID());
        }
        if ($this->client_params->getOID() !== null && $this->client_params->getOID() !== 0) {
            $this->masterkey->setOid($this->client_params->getOID());
        }
        if ($this->client_params->getSource() !== null && $this->client_params->getSource() !== '' && $this->client_params->getSource() !== '0') {
            $this->masterkey->setSource($this->client_params->getSource());
        }

        $lead = new Lead();

        if ($this->form_data->getBrandID() !== null) {
            $lead->setBrandId($this->form_data->getBrandID());
            $this->masterkey->setBrand($this->form_data->getBrandID());
        } else {
            $lead->setBrandId($this->getBrandID());
        }

        $lead->setDisclaimerId($this->getDisclaimerID());

        if ($this->form_data->getCTA() != null) {
            $lead->setType($this->form_data->getCTA());
        } else {
            $lead->setType($this->getCTA()[0]);
        }

        $this->masterkey->find($lead->getType());
        $masterkey = $this->masterkey->getMasterkey();

        $lead->setMasterkey($masterkey);

        if ($this->form_data->getFirstName()) {
            $lead->setFirstName($this->form_data->getFirstName());
        }
        if ($this->form_data->getLastName()) {
            $lead->setLastName($this->form_data->getLastName());
        }
        if ($this->form_data->getCompany()) {
            $lead->setCompany($this->form_data->getCompany());
        }
        if ($this->form_data->getAddress()) {
            $lead->setAddress($this->form_data->getAddress());
        }
        if ($this->form_data->getZipCode()) {
            $lead->setZipCode($this->form_data->getZipCode());
        }
        if ($this->form_data->getCity()) {
            $lead->setCity($this->form_data->getCity());
        }
        if ($this->form_data->getEmail()) {
            $lead->setEmail($this->form_data->getEmail());
        }
        if ($this->form_data->getPhone()) {
            $lead->setPhoneNumber($this->form_data->getPhone());
        }
        if ($this->form_data->getDealerSincom()) {
            $lead->setDealerSincom($this->form_data->getDealerSincom());
        }
        if ($this->form_data->getModelID()) {
            $lead->setModelId($this->form_data->getModelID());
        }
        if ($this->form_data->getAppointmentDate()) {
            $lead->setAppointmentDate($this->form_data->getAppointmentDate());
        }
        if ($this->form_data->getRelatedID()) {
            $lead->setRelatedId($this->form_data->getRelatedID());
        }
        if ($this->form_data->getOrigin()) {
            // Magiczne dodanie parametrów z query string do origin'a
            // lepszego miejsca nie znalazłem

            // Pobranie parametrów zapisanych w sesji
            $client_session_params = $this->getClientParams()->getAllParams();
            $client_session_params = array_filter($client_session_params);
            // Dostarczony origin
            $origin = $this->form_data->getOrigin();
            // Sprawdzenie, czy dostarczony origin jest prawidłowym url
            if (filter_var($origin, FILTER_VALIDATE_URL) !== false) {
                // Sytuacja, gdy url ma ogonek - wyciągamy tylko ogonek jako string
                if ($query_string = parse_url((string) $origin, PHP_URL_QUERY)) {
                    // ogonek z stringa zamieniamy na tablice
                    parse_str($query_string, $query_string_as_array);
                    // Łączymy parametry z stringa z dostarczonymi z sesji
                    $client_session_params = array_merge($query_string_as_array, $client_session_params);
                    // Budujemy url (usuwamy query z oryginalnego origin'a) i doklejamy wszystkie inne parametry
                    $origin = str_replace($query_string, '', $origin) . http_build_query($client_session_params);
                } else {
                    // URl nie ma ogona dodajemy tylko parametry z sesji
                    $origin = $origin .'?'. http_build_query($client_session_params);
                }
            } else {
                // Gdy origin nie jest url - np sztuczne generowany origin - dodajemy tylko parametry z sesji
                $origin = $origin .'?'. http_build_query($client_session_params);
            }
            $lead->setOrigin($origin);
        }
        if ($this->form_data->getUserAgent()) {
            $lead->setUserAgent($this->form_data->getUserAgent());
        }
        if ($this->form_data->getDevice()) {
            $lead->setDevice($this->form_data->getDevice());
        }
        if ($this->form_data->getIP()) {
            $lead->setIp($this->form_data->getIP());
        }
        if ($this->form_data->getComment()) {
            $lead->setComment($this->form_data->getComment());
        }
        if ($this->form_data->getVin()) {
            $lead->setVin($this->form_data->getVin());
        }

        if ($this->form_data->getEnergyType()) {
            $lead->setEnergyType($this->form_data->getEnergyType());
        }

        if ($this->form_data->getCommunicationChannel()) {
            $lead->setCommunicationChannel($this->form_data->getCommunicationChannel());
        }

        if ($this->skipCallCenterVerification) {
            $lead->skipCallCenterVerification(true);
        }

        $map_agreement_fields = $this->getDisclaimer()->getAgreements()->agreementsMapper();

        foreach ($map_agreement_fields as $map_agreement_field => $map_agreement_field_privacy) {
            foreach ($map_agreement_field_privacy as $privacy_key) {
                if ($this->form_data->getAgreementByKey($map_agreement_field) == 1) {
                    $lead->setPrivacy($privacy_key, Lead::PRIVACY_ACCEPTED);
                } elseif ($this->form_data->getAgreementByKey($map_agreement_field) == 0) {
                    $lead->setPrivacy($privacy_key, Lead::PRIVACY_REFUSED);
                } else {
                    $lead->setPrivacy($privacy_key, Lead::PRIVACY_UNKNOWN);
                }
            }
        }

        $create = new Create(
            $fcaapiAuth->getApiUrl(),
            $fcaapiAuth->getApiUser(),
            $fcaapiAuth->getApiPass()
        );

        $create->setLead($lead);

        // Jeśli zapis lead się nie uda, zwracamy wyjątek i w aplikacji go obsługujemy.
        $create->execute();

        $return_data = [
            'status'       => 'success',
            'lead_id'      => $lead->getId(),
            'cta'          => $lead->getType(),
            'brand'        => $lead->getBrandId(),
            'masterkey_id' => $masterkey->getId(),
        ];

        if ($debug_saved_data) {
            $return_data['lead_data'] = $lead->getProperties();
            $return_data['masterkey'] = [
               'id' => $masterkey->getId(),
               'brand' => $masterkey->getBrand(),
               'cta' => $masterkey->getType(),
               'cid' => $masterkey->getCid(),
               'oid' => $masterkey->getOid(),
               'source' => $masterkey->getSource(),
            ];
        }

        return $return_data;
    }

    public function regenerateCsrfToken(): string
    {
        /** @var FieldInterface $field */
        foreach ($this->fields as $key => $field) {
            if ($field->getFieldClassName() === CsrfField::class) {
                return $this->fields[$key]->regenerateCsrfValueToken();
            }
        }
        throw new FormsLIBException('CSFR token is not supported for this form');
    }

    public function skipCallCenterVerification(bool $flag = true): void
    {
        $this->skipCallCenterVerification = $flag;
    }
}
