<?php
declare(strict_types=1);

namespace FCAPoland\FormsLib;

use Exception;
use FCAPoland\ApiPrivacyHelper\PrivacyPolicy;
use FCAPoland\ApiPrivacyHelper\StorageInterface;
use FCAPoland\FormsLib\Config\LibConfig;
use FCAPoland\FormsLib\Exception\FormNotFoundException;
use FCAPoland\FormsLib\Exception\FormsNotFoundException;
use FCAPoland\FormsLib\Exception\InvalidArgumentException;
use FCAPoland\FormsLib\Form\Form;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;

/**
 * Class FormManager
 * @package FCAPoland\FormsLib
 */
class FormManager
{
    use LoggerAwareTrait;

    private \FCAPoland\ApiPrivacyHelper\StorageInterface $storage;

    private ?\FCAPoland\FormsLib\Config\LibConfig $libConfig = null;

    private ?\Psr\SimpleCache\CacheInterface $cache = null;

    private ?array $forms = null;

    public function __construct()
    {
        if (session_status() != PHP_SESSION_ACTIVE) {
            session_start();
        }
    }

    public function setLibConfig(LibConfig $libConfig): void
    {
        $this->libConfig = $libConfig;
        $this->storage = new Storage($this->libConfig->getStoragePath());
        $this->loadForms();
    }

    public function setCache(CacheInterface $cache): void
    {
        $this->cache = $cache;
    }

    public function getFormByID($id): Form
    {
        if (!isset($this->forms[$id])) {
            throw new FormNotFoundException(sprintf('Form %s not found', $id));
        }
        return $this->forms[$id];
    }

    private function loadForms(): void
    {
        $this->forms = [];
        if ($this->libConfig->getForms() and is_array($this->libConfig->getForms())) {
            foreach ($this->libConfig->getForms() as $form_config) {
                try {
                    $form = new Form();
                    if ($this->cache instanceof CacheInterface) {
                        $form->setCache($this->cache);
                    }
                    if ($this->logger instanceof LoggerInterface) {
                        $form->setLogger($this->logger);
                    }
                    $form->setStorage($this->getStorage());
                    $form->setApiBaseUrl($this->libConfig->getApiBaseUrl());
                    $form->initForm($form_config);
                    $this->forms[$form->getFormID()] = $form;
                } catch (InvalidArgumentException $e) {
                    throw new Exception($e->getMessage(), $e->getCode(), $e);
                }
            }
        } else {
            throw new FormsNotFoundException('Forms not found. Please Check lib config');
        }
    }

    public function getPrivacyPolicy(): ?string
    {
        try {
            $privacyPolicy = new PrivacyPolicy($this->getStorage());
            $privacyPolicy->setApiBaseUrl($this->libConfig->getApiBaseUrl());
            if ($privacy_policy_hash = $this->libConfig->getPrivacyPolicyHash()) {
                $privacyPolicy->setPrivacyPolicyHash($privacy_policy_hash);
            }
            if ($this->cache instanceof CacheInterface) {
                $privacyPolicy->setCache($this->cache);
            }
            return $privacyPolicy->getPrivacyPolicyContent() ?? null;
        } catch (Exception $exception) {
            if ($this->logger instanceof LoggerInterface) {
                $this->logger->error('Can not init PrivacyPolicy object: ' . $exception->getMessage());
            }
            throw new Exception('Can not get privacy', $exception->getCode(), $exception);
        }
    }

    public function getStorage(): StorageInterface
    {
        return $this->storage;
    }

    public function setStorage(StorageInterface $storage): void
    {
        $this->storage = $storage;
    }
}
