<?php


namespace Unit;


use FCAPoland\FormsLib\Field\BaseField;
use FCAPoland\FormsLib\Field\FieldInterface;
use FCAPoland\FormsLib\Field\Rule\BaseRule;
use FCAPoland\FormsLib\Field\Rule\CheckRecaptcha;
use FCAPoland\FormsLib\Field\Rule\InArray;
use FCAPoland\FormsLib\Field\Rule\IsEmail;
use FCAPoland\FormsLib\Field\Rule\IsPhone;
use FCAPoland\FormsLib\Field\Rule\MaxLength;
use FCAPoland\FormsLib\Field\Rule\MinLength;
use FCAPoland\FormsLib\Field\Rule\NotEmpty;
use FCAPoland\FormsLib\Field\Rule\RuleInterface;
use FCAPoland\FormsLib\Field\Rule\ValidDate;
use FCAPoland\FormsLib\Field\Rule\ValidDealerSincom;
use FCAPoland\FormsLib\Field\Rule\ValidPostCode;
use PHPUnit\Framework\TestCase;

/**
 * Class RulesTest
 * @package Unit
 */
class RulesTest extends TestCase
{
    public function testBaseRule(): void
    {
        $notEmpty = new NotEmpty(new BaseField());
        $this->assertInstanceOf(RuleInterface::class, $notEmpty);
        $notEmpty->setRegex('(.*)');
        $this->assertEquals('(.*)', $notEmpty->getRegex());
        $this->assertTrue($notEmpty->isValidValue('A'));
        $notEmpty->setRegex('/TEST\-(.*)/');
        $this->assertTrue($notEmpty->isValidValue('TEST-A'));
        $this->assertTrue($notEmpty->isValidValue('TEST-B'));
        $this->assertFalse($notEmpty->isValidValue('1TEST'));
        $this->assertEquals(NotEmpty::class, $notEmpty->getRuleName());
        $this->assertInstanceOf(FieldInterface::class, $notEmpty->getField());
        $this->assertEquals(BaseField::class, $notEmpty->getFieldClassName());

       // $this->expectException(TypeError::class);
        $this->assertFalse($notEmpty->isValidValue([]));
    }

    public function testCheckRecaptcha(): void
    {
        $checkRecaptcha = new CheckRecaptcha(new BaseField(), 'site_key', 'secret_key');
        $this->assertInstanceOf(RuleInterface::class, $checkRecaptcha);
        $this->assertEquals('site_key', $checkRecaptcha->getSiteKey());
        $this->assertEquals('secret_key', $checkRecaptcha->getSecretKey());
        // TODO do przerobienia aby nie łączyło się z zew serwerami
        // Sprawdzane jest w google czy kod jest prawidłowy dla fake danych
        $this->assertFalse($checkRecaptcha->isValidValue('xxx'));
        // TODO dorobienie poprawnej walidacji - LVL_UP!
        // Czyli jakaś symulacja recaptch'y
    }

    public function testInArray(): void
    {
        $inArray = new InArray(new BaseField(), ['111']);
        $this->assertInstanceOf(RuleInterface::class, $inArray);
        $this->assertFalse($inArray->isValidValue('222'));
        $this->assertTrue($inArray->isValidValue(111));
        $this->assertTrue($inArray->isValidValue('111'));
        $this->assertFalse($inArray->isValidValue(['111']));
    }

    public function testIsEmail(): void
    {
        $isEmail = new IsEmail(new BaseField());
        $this->assertInstanceOf(RuleInterface::class, $isEmail);
        $this->assertFalse($isEmail->isValidValue('222'));
        $this->assertTrue($isEmail->isValidValue('test@test.com'));
        $this->assertTrue($isEmail->isValidValue('test@test.com.pl'));
        $this->assertTrue($isEmail->isValidValue('test.test@test.com.pl'));
    }

    public function testIsPhone(): void
    {
        $isPhone = new IsPhone(new BaseField());
        $this->assertInstanceOf(RuleInterface::class, $isPhone);
        $this->assertFalse($isPhone->isValidValue('222'));
        $this->assertFalse($isPhone->isValidValue('6666666666'));
        $this->assertFalse($isPhone->isValidValue('+44 666-666-666'));
        $this->assertTrue($isPhone->isValidValue('666-666-666'));
        $this->assertTrue($isPhone->isValidValue('666666666'));
        $this->assertTrue($isPhone->isValidValue('+48 666-666-666'));
        $this->assertTrue($isPhone->isValidValue('+48666666666'));
    }

    public function testMaxLength(): void
    {
        $maxLength = new MaxLength(new BaseField(), 2);
        $this->assertInstanceOf(RuleInterface::class, $maxLength);
        $this->assertFalse($maxLength->isValidValue('222'));
        $this->assertFalse($maxLength->isValidValue('6666666666'));
        $this->assertFalse($maxLength->isValidValue('abc'));
        $this->assertTrue($maxLength->isValidValue(''));
        $this->assertTrue($maxLength->isValidValue('1'));
        $this->assertTrue($maxLength->isValidValue('ab'));
    }

    public function testMinLength(): void
    {
        $minLength = new MinLength(new BaseField(), 2);
        $this->assertInstanceOf(RuleInterface::class, $minLength);
        $this->assertFalse($minLength->isValidValue('1'));
        $this->assertFalse($minLength->isValidValue(''));
        $this->assertTrue($minLength->isValidValue('ab'));
        $this->assertTrue($minLength->isValidValue('aaa'));
    }

    public function testNoEmpty(): void
    {
        $notEmpty = new NotEmpty(new BaseField());
        $this->assertInstanceOf(RuleInterface::class, $notEmpty);
        $this->assertFalse($notEmpty->isValidValue(''));
        $this->assertFalse($notEmpty->isValidValue(null));
        $this->assertFalse($notEmpty->isValidValue([]));
        $this->assertTrue($notEmpty->isValidValue('0'));
        $this->assertTrue($notEmpty->isValidValue('ab'));
        $this->assertTrue($notEmpty->isValidValue('aaa'));
    }

    public function testValidDate(): void
    {
        $validDate = new ValidDate(new BaseField());
        $this->assertInstanceOf(RuleInterface::class, $validDate);
        $this->assertFalse($validDate->isValidValue('99-12-2017'));
        $this->assertFalse($validDate->isValidValue('12-99-2017'));
        $this->assertFalse($validDate->isValidValue('12-99-18'));
        $this->assertFalse($validDate->isValidValue('2020-01-01'));
        $this->assertTrue($validDate->isValidValue('02-12-2020'));
        $this->assertTrue($validDate->isValidValue('05-01-2020'));
    }

    public function testDealerSincom(): void
    {
        $validDealerSincom = new ValidDealerSincom(new BaseField());
        $this->assertInstanceOf(RuleInterface::class, $validDealerSincom);
        $this->assertFalse($validDealerSincom->isValidValue(''));
        $this->assertFalse($validDealerSincom->isValidValue('666666666'));
        $this->assertFalse($validDealerSincom->isValidValue('6666666.5666'));
        $this->assertTrue($validDealerSincom->isValidValue('1111111.000'));
    }

    public function testPostCode(): void
    {
        $validPostCode = new ValidPostCode(new BaseField());
        $this->assertInstanceOf(RuleInterface::class, $validPostCode);
        $this->assertFalse($validPostCode->isValidValue(''));
        $this->assertFalse($validPostCode->isValidValue('83000'));
        $this->assertFalse($validPostCode->isValidValue('83 000'));
        $this->assertTrue($validPostCode->isValidValue('83-000'));
    }
}
