<?php
declare(strict_types=1);

namespace FCAPoland\LeadsAPIHelper\Request;

use FCAPoland\LeadsAPIHelper\Exception\APIException;
use FCAPoland\LeadsAPIHelper\Exception\LeadsAPIHelperException;
use FCAPoland\LeadsAPIHelper\Lead;
use FCAPoland\LeadsAPIHelper\Request;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;

class Create extends Request
{
    use LoggerAwareTrait;

    // Max number of retry after server errors
    const MAX_RETRY = 1;

    const QUERY_PARAM_SKIP_CALLCENTER = 'skip-callcenter-verification';

    /**
     * @var Lead
     */
    private $lead;

    protected $method = self::METHOD_PUT;

    /**
     * @var int
     */
    private $retry_counter = 0;

    /**
     * @return string
     * @throws LeadsAPIHelperException
     */
    protected function createBody(): string
    {
        $body = json_encode($this->lead->getProperties());
        if ($body === false) {
            $this->logMalformedData($this->lead->getProperties());
            $body = json_encode($this->lead->getProperties(), JSON_INVALID_UTF8_SUBSTITUTE);
            if ($body === false) {
                throw new LeadsAPIHelperException(
                    'Can not encode properties of lead, error: ' . json_last_error_msg()
                );
            }
        }
        return $body;
    }

    public function setLead(Lead $lead)
    {
        $this->lead = $lead;
    }

    /**
     * @throws LeadsAPIHelperException
     */
    public function execute()
    {
        if (!($this->lead instanceof Lead)) {
            throw new LeadsAPIHelperException(
                'Unable to execute lead create request with no lead data provided.' .
                ' Please attach a lead to the request.'
            );
        }

        if ($this->lead->getSkipCallCenterVerification()) {
            $this->setQueryParam(self::QUERY_PARAM_SKIP_CALLCENTER, true);
        }

        try {
            parent::execute();
        } catch (APIException $api_exception) {
            // For server errors try again execute saving lead in FCA API
            if ($api_exception->getCode() >= 500) {
                if ($this->retry_counter < self::MAX_RETRY) {
                    $this->retry_counter++;
                    // Waiting time before execute again request
                    sleep(10);
                    $this->execute();
                }
            }
            throw new LeadsAPIHelperException($api_exception->getMessage(), $api_exception->getCode());
        }

        $response_body = $this->getResponseBody();

        if (201 === $this->getResponseCode() and is_numeric($response_body)) {
            // Remember ID of newly created lead
            $this->lead->setId(intval($response_body));
        } else {
            throw new LeadsAPIHelperException('Unrecognized error');
        }
    }

    /**
     * @param array $props
     */
    private function logMalformedData(array $props): void
    {
        $list = [];
        foreach ($props as $field => $value) {
            if (strpos(json_encode($value, JSON_INVALID_UTF8_SUBSTITUTE), "\ufffd") !== false) {
                $list[$field] = $value;
            }
        }
        if (count($list) > 0) {
            if ($this->logger instanceof LoggerInterface) {
                $this->logger->debug(
                    'Tried to send lead with malformed UTF-8 characters,' .
                    ' detected incorrect properties with data: ' . json_encode($list, JSON_INVALID_UTF8_SUBSTITUTE)
                );
                $this->logger->warning(
                    'Could not encode full properties properly to JSON, ' .
                    'invalid chars will be substituted with \0xfffd (Unicode Character "REPLACEMENT CHARACTER"). ' .
                    'Incorrect field/s : ' . implode(', ', array_keys($list))
                );
            }
        } else {
            $this->logger->warning(
                'Could not encode full properties properly to JSON, but can not find incorrect chars.'
            );
        }
    }
}
