<?php
declare(strict_types=1);

namespace FCAPoland\LeadsAPIHelper;

use FCAPoland\LeadsAPIHelper\Exception\InvalidArgumentException;
use FCAPoland\LeadsAPIHelper\Exception\LeadsAPIHelperException;
use Psr\Log\LoggerAwareTrait;
use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;

/**
 * Class MasterkeyFinder
 * @package FCAPoland\LeadsAPIHelper
 */
class MasterkeyFinder
{
    use LoggerAwareTrait;

    const API_URL = 'https://api.fcapoland.pl/masterkey/find?';
    const CACHE_KEY = 'masterkey_finder:';
    const CACHE_TTL = 60 * 60 * 72; // 72H

    /**
     * @var CacheInterface
     */
    private $cache;

    /**
     * @var string
     */
    private $brand;

    /**
     * @var string
     */
    private $type;

    /**
     * @var integer|null
     */
    private $cid;

    /**
     * @var integer|null
     */
    private $oid;

    /**
     * @var string|null
     */
    private $source;

    /**
     * @var string|null
     */
    private $default_source;

    /**
     * @var integer
     */
    private $app_cid;

    /**
     * @var integer|null
     */
    private $app_oid;

    /**
     * @var string
     */
    private $app_source;

    /**
     * @var Masterkey
     */
    private $masterkey;

    /**
     * @return string
     */
    private function getCacheKey(): string
    {
        return md5(self::CACHE_KEY . $this->getParamsToFetch());
    }

    /**
     * @param $type
     * @return MasterkeyFinder
     * @throws LeadsAPIHelperException
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function find($type): MasterkeyFinder
    {
        $this->type = $type;
        if (empty($this->brand) or empty($type) or empty($this->app_cid)) {
            throw new LeadsAPIHelperException('Masterkey can not be find without brand, type and campaign id!');
        }
        if ($this->cache instanceof CacheInterface) {
            $masterkey_json = $this->cache->get($this->getCacheKey());
            if (!$masterkey_json) {
                try {
                    $masterkey_json = $this->fetch();
                } catch (LeadsAPIHelperException $e) {
                    throw new LeadsAPIHelperException('Masterkey can not be find in API');
                }
                if ($this->logger instanceof LoggerInterface) {
                    $this->logger->info(
                        '[MASTERKEY FINDER] - Get MK for ' . $this->getCacheKey() .
                        ' from API because do not found in cache: ' . $masterkey_json
                    );
                }
                $this->cache->set($this->getCacheKey(), $masterkey_json, self::CACHE_TTL);
            } else {
                if ($this->logger instanceof LoggerInterface) {
                    $this->logger->info(
                        '[MASTERKEY FINDER] - Get MK for ' . $this->getCacheKey() . ' from cache: ' . $masterkey_json
                    );
                }
            }
        } else {
            try {
                $masterkey_json = $this->fetch();
                if ($this->logger instanceof LoggerInterface) {
                    $this->logger->info(
                        '[MASTERKEY FINDER] - Get MK for ' . $this->getCacheKey() .
                        ' from API because cache is disabled: ' . $masterkey_json
                    );
                }
            } catch (LeadsAPIHelperException $e) {
                throw new LeadsAPIHelperException('Masterkey can not be find in API');
            }
        }
        if (!$this->isJson($masterkey_json)) {
            throw new LeadsAPIHelperException('Masterkey found in API but is invalid json format!');
        }

        $masterkey = json_decode($masterkey_json, true);

        if (!array_key_exists('id', $masterkey) or
            !array_key_exists('brand_id', $masterkey) or
            !array_key_exists('type', $masterkey) and
            !array_key_exists('campaign_id', $masterkey) or
            !array_key_exists('offer_id', $masterkey) or
            !array_key_exists('source', $masterkey)
        ) {
            throw new LeadsAPIHelperException('Masterkey found in API but is invalid format!');
        }

        if ($this->getCid() != $masterkey['campaign_id']) {
            if ($this->logger instanceof LoggerInterface) {
                $this->logger->info(
                    '[MASTERKEY FINDER] - Received MK with CID :CID but GET param is :GET_CID',
                    [':GET_CID' => ($this->getCid() ?? 'EMPTY'), ':CID' => $masterkey['campaign_id']]
                );
            }
        }
        if ($this->getOid() != $masterkey['offer_id']) {
            if ($this->logger instanceof LoggerInterface) {
                if ($this->logger instanceof LoggerInterface) {
                    $this->logger->info(
                        '[MASTERKEY FINDER] - Received MK with OID :OID but GET param is :GET_OID',
                        [':GET_OID' => ($this->getOid() ?? 'EMPTY'), ':OID' => $masterkey['offer_id']]
                    );
                }
            }
        }
        if ($this->getSource() != $masterkey['source']) {
            if ($this->logger instanceof LoggerInterface) {
                $this->logger->info(
                    '[MASTERKEY FINDER] - Received MK with SOURCE :SOURCE but GET param is :GET_SOURCE',
                    [':GET_SOURCE' => ($this->getSource() ?? 'EMPTY'), ':SOURCE' => $masterkey['source']]
                );
            }
        }


        $this->setMasterkey(new Masterkey(
            (int) $masterkey['id'],
            (string) $masterkey['brand_id'],
            (string) $masterkey['type'],
            (int) $masterkey['campaign_id'],
            is_numeric($masterkey['offer_id']) ? ((int) $masterkey['offer_id']) : null,
            (string) $masterkey['source']
        ));
        return $this;
    }

    /**
     * @return Masterkey
     * @throws LeadsAPIHelperException
     */
    public function getMasterkey(): Masterkey
    {
        if (!$this->masterkey instanceof Masterkey) {
            throw new LeadsAPIHelperException('Masterkey not loaded use before get find() function!');
        }
        return $this->masterkey;
    }


    /**
     * @param mixed $cache
     * @return MasterkeyFinder
     */
    public function setCache(CacheInterface $cache): MasterkeyFinder
    {
        $this->cache = $cache;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getBrand(): string
    {
        return $this->brand;
    }

    /**
     * @param mixed $brand
     * @return MasterkeyFinder
     */
    public function setBrand($brand): MasterkeyFinder
    {
        $this->brand = $brand;
        return $this;
    }

    /**
     * @return int|null
     */
    public function getCid(): ?int
    {
        return $this->cid;
    }

    /**
     * @param mixed $cid
     * @return MasterkeyFinder
     */
    public function setCid($cid): MasterkeyFinder
    {
        if ($cid == null or $cid == '' or !is_numeric($cid)) {
            $cid = null;
        }
        $this->cid = $cid;
        return $this;
    }

    /**
     * @return mixed|null
     */
    public function getOid(): ?int
    {
        return $this->oid;
    }

    /**
     * @param mixed $oid
     * @return MasterkeyFinder
     */
    public function setOid($oid): MasterkeyFinder
    {
        if ($oid == null or $oid == '' or !is_numeric($oid)) {
            $oid = null;
        }
        $this->oid = $oid;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getSource(): ?string
    {
        return $this->source;
    }

    /**
     * @param mixed $source
     * @return MasterkeyFinder
     */
    public function setSource($source): MasterkeyFinder
    {
        $this->source = $source;
        return $this;
    }

    /**
     * @return string|null
     */
    public function getDefaultSource(): ?string
    {
        return $this->default_source;
    }

    /**
     * @param mixed $default_source
     * @return MasterkeyFinder
     */
    public function setDefaultSource($default_source): MasterkeyFinder
    {
        $this->default_source = $default_source;
        return $this;
    }

    /**
     * @return int
     */
    public function getAppCid(): int
    {
        return $this->app_cid;
    }

    /**
     * @param mixed $app_cid
     * @return MasterkeyFinder
     * @throws InvalidArgumentException
     */
    public function setAppCid($app_cid): MasterkeyFinder
    {
        if ($app_cid == null or $app_cid == '') {
            return $this;
        }
        if (!is_numeric($app_cid)) {
            throw new InvalidArgumentException('AppCid must be specified using int, ' . gettype($app_cid) . ' given');
        }
        $this->app_cid = $app_cid;
        return $this;
    }

    /**
     * @return int|null
     */
    public function getAppOid(): ?int
    {
        return $this->app_oid;
    }

    /**
     * @param mixed $app_oid
     * @return MasterkeyFinder
     * @throws InvalidArgumentException
     */
    public function setAppOid($app_oid): MasterkeyFinder
    {
        if ($app_oid == null or $app_oid == '') {
            return $this;
        }
        if (!is_numeric($app_oid)) {
            throw new InvalidArgumentException(
                'AppOid must be specified using int, ' . gettype($app_oid) . ' given'
            );
        }
        $this->app_oid = $app_oid;
        return $this;
    }

    /**
     * @return string
     */
    public function getAppSource(): string
    {
        return $this->app_source;
    }

    /**
     * @param string $app_source
     * @return MasterkeyFinder
     */
    public function setAppSource(string $app_source): MasterkeyFinder
    {
        $this->app_source = $app_source;
        return $this;
    }

    /**
     * @return string
     */
    public function getType(): string
    {
        return $this->type;
    }

    /**
     * @param Masterkey $masterkey
     * @return void
     */
    private function setMasterkey(Masterkey $masterkey): void
    {
        $this->masterkey = $masterkey;
    }

    /**
     * @return string
     */
    private function getParamsToFetch(): string
    {
        $params = [
            'brand_id' => $this->getBrand(),
            'type' => $this->getType(),
            'campaign_id' => $this->getCid(),
            'offer_id' => $this->getOid(),
            'source' => $this->getSource(),
            'default_source' => $this->getDefaultSource(),
            'app_campaign_id' => $this->getAppCid(),
            'app_offer_id' => $this->getAppOid(),
            'app_source' => $this->getAppSource(),
        ];
        return ((http_build_query($params)));
    }

    /**
     * @return bool|string
     * @throws LeadsAPIHelperException
     */
    private function fetch()
    {
        try {
            $curl_handle = curl_init();
            curl_setopt($curl_handle, CURLOPT_URL, static::API_URL . $this->getParamsToFetch());
            curl_setopt($curl_handle, CURLOPT_CONNECTTIMEOUT, 15);
            curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl_handle, CURLOPT_USERAGENT, 'FCA LeadHelper - MasterKey Finder');
            $query = curl_exec($curl_handle);
            $http_code = curl_getinfo($curl_handle, CURLINFO_HTTP_CODE);
            curl_close($curl_handle);

            if ($query === false or $http_code >= 400) {
                throw new LeadsAPIHelperException(
                    'Unable to get masterkey from FCA API for data: ' . $this->getParamsToFetch()
                );
            }

            return $query;
        } catch (LeadsAPIHelperException $e) {
            throw new LeadsAPIHelperException($e->getMessage(), null, $e);
        }
    }

    /**
     * @param $string
     * @return bool
     */
    private function isJson($string): bool
    {
        json_decode($string);
        return (json_last_error() == JSON_ERROR_NONE);
    }
}
