<?php

namespace FCAPoland\LeadsAPIHelper;

use Psr\Log\LoggerInterface;
use Psr\SimpleCache\CacheInterface;

/**
 * Class MasterkeyFinder
 * @package FCAPoland\LeadsAPIHelper
 */
class MasterkeyFinder
{

    const API_URL = 'https://api.fcapoland.pl/masterkey/find?';
    const CACHE_KEY = 'masterkey_finder:';
    const CACHE_TTL = 60 * 60 * 72; // 72H

    /**
     * @var CacheInterface
     */
    private $cache;

    /**
     * @var integer
     */
    private $brand;

    /**
     * @var string
     */
    private $type;

    /**
     * @var integer
     */
    private $cid;

    /**
     * @var integer
     */
    private $oid;

    /**
     * @var string
     */
    private $source;

    /**
     * @var string
     */
    private $default_source;

    /**
     * @var integer
     */
    private $app_cid;

    /**
     * @var integer
     */
    private $app_oid;

    /**
     * @var string
     */
    private $app_source;

    /**
     * @var Masterkey
     */
    private $masterkey;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @return string
     */
    private function getCacheKey()
    {
        return self::CACHE_KEY . $this->getParamsToFetch();
    }

    /**
     * @param $type
     * @return MasterkeyFinder
     * @throws Exception
     * @throws \Psr\SimpleCache\InvalidArgumentException
     */
    public function find($type)
    {
        $this->type = $type;
        if (empty($this->brand) or empty($type) or empty($this->app_cid)) {
            throw new Exception('Masterkey can not be find without brand, type and cammpaign id!');
        }
        if ($this->cache instanceof CacheInterface) {
            $masterkey_json = $this->cache->get($this->getCacheKey());
            if (!$masterkey_json) {
                try {
                    $masterkey_json = $this->fetch();
                } catch (Exception $e) {
                    throw new Exception('Masterkey can not be find in API');
                }
                if ($this->logger) {
                    $this->logger->info('[MASTERKEY FINDER] - Get MK for ' . $this->getCacheKey() . ' from API because do not found in cache: ' . $masterkey_json);
                }
                $this->cache->set($this->getCacheKey(), $masterkey_json, self::CACHE_TTL);
            } else {
                if ($this->logger) {
                    $this->logger->info('[MASTERKEY FINDER] - Get MK for ' . $this->getCacheKey() . ' from cache: ' . $masterkey_json);
                }
            }
        } else {
            try {
                $masterkey_json = $this->fetch();
                if ($this->logger) {
                    $this->logger->info('[MASTERKEY FINDER] - Get MK for ' . $this->getCacheKey() . ' from API because cache is disabled: ' . $masterkey_json);
                }
            } catch (Exception $e) {
                throw new Exception('Masterkey can not be find in API');
            }
        }
        if (!$this->isJson($masterkey_json)) {
            throw new Exception('Masterkey found in API but is invalid json format!');
        }

        $masterkey = json_decode($masterkey_json, true);

        if (!array_key_exists('id', $masterkey) or
            !array_key_exists('brand_id', $masterkey) or
            !array_key_exists('type', $masterkey) and
            !array_key_exists('campaign_id', $masterkey) or
            !array_key_exists('offer_id', $masterkey) or
            !array_key_exists('source', $masterkey)
        ) {
            throw new Exception('Masterkey found in API but is invalid format!');
        }


        if ($this->getCid() != $masterkey['campaign_id']) {
            $this->logger->info(
                '[MASTERKEY FINDER] - Received MK with CID :CID but GET param is :GET_CID',
                [':GET_CID' => ($this->getCid() ? $this->getCid() : 'EMPTY'), ':CID' => $masterkey['campaign_id']]
            );
        }
        if ($this->getOid() != $masterkey['offer_id']) {
            $this->logger->info(
                '[MASTERKEY FINDER] - Received MK with OID :OID but GET param is :GET_OID',
                [':GET_OID' => ($this->getOid() ? $this->getOid() : 'EMPTY'), ':OID' => $masterkey['offer_id']]
            );
        }
        if ($this->getSource() != $masterkey['source']) {
            $this->logger->info(
                '[MASTERKEY FINDER] - Received MK with SOURCE :SOURCE but GET param is :GET_SOURCE',
                [':GET_SOURCE' => ($this->getSource() ? $this->getSource() : 'EMPTY'), ':SOURCE' => $masterkey['source']]
            );
        }


        $this->setMasterkey(new Masterkey(
            $masterkey['id'],
            $masterkey['brand_id'],
            $masterkey['type'],
            $masterkey['campaign_id'],
            $masterkey['offer_id'],
            $masterkey['source']
        ));
        return $this;
    }

    /**
     * @return Masterkey
     * @throws Exception
     */
    public function getMasterkey()
    {
        if (!$this->masterkey instanceof Masterkey) {
            throw new Exception('Masterkey not loaded use before get find() function!');
        }
        return $this->masterkey;
    }


    /**
     * @param mixed $cache
     * @return MasterkeyFinder
     */
    public function setCache(CacheInterface $cache)
    {
        $this->cache = $cache;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getBrand()
    {
        return $this->brand;
    }

    /**
     * @param mixed $brand
     * @return MasterkeyFinder
     */
    public function setBrand($brand)
    {
        $this->brand = $brand;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getCid()
    {
        return $this->cid;
    }

    /**
     * @param mixed $cid
     * @return MasterkeyFinder
     */
    public function setCid($cid)
    {
        if ($cid == null or $cid == '' or !is_numeric($cid)) {
            $cid = null;
        }
        $this->cid = $cid;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getOid()
    {
        return $this->oid;
    }

    /**
     * @param mixed $oid
     * @return MasterkeyFinder
     */
    public function setOid($oid)
    {
        if ($oid == null or $oid == '' or !is_numeric($oid)) {
            $oid = null;
        }
        $this->oid = $oid;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getSource()
    {
        return $this->source;
    }

    /**
     * @param mixed $source
     * @return MasterkeyFinder
     */
    public function setSource($source)
    {
        $this->source = $source;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getDefaultSource()
    {
        return $this->default_source;
    }

    /**
     * @param mixed $default_source
     * @return MasterkeyFinder
     */
    public function setDefaultSource($default_source)
    {
        $this->default_source = $default_source;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getAppCid()
    {
        return $this->app_cid;
    }

    /**
     * @param mixed $app_cid
     * @return MasterkeyFinder
     * @throws InvalidArgumentException
     */
    public function setAppCid($app_cid)
    {
        if ($app_cid == null or $app_cid == '') {
            return $this;
        }
        if (!is_numeric($app_cid)) {
            throw new InvalidArgumentException('AppCid must be specified using int, ' . gettype($app_cid) . ' given');
        }
        $this->app_cid = $app_cid;
        return $this;
    }

    /**
     * @return mixed
     */
    public function getAppOid()
    {
        return $this->app_oid;
    }

    /**
     * @param mixed $app_oid
     * @return MasterkeyFinder
     * @throws InvalidArgumentException
     */
    public function setAppOid($app_oid)
    {
        if ($app_oid == null or $app_oid == '') {
            return $this;
        }
        if (!is_numeric($app_oid)) {
            throw new InvalidArgumentException('AppOid must be specified using int, ' . gettype($app_oid) . ' given');
        }
        $this->app_oid = $app_oid;
        return $this;
    }

    /**
     * @return string
     */
    public function getAppSource()
    {
        return $this->app_source;
    }

    /**
     * @param string $app_source
     * @return MasterkeyFinder
     */
    public function setAppSource($app_source)
    {
        $this->app_source = $app_source;
        return $this;
    }

    /**
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * @param Masterkey $masterkey
     * @return MasterkeyFinder
     */
    private function setMasterkey(Masterkey $masterkey)
    {
        $this->masterkey = $masterkey;
        return $this;
    }

    /**
     * @return string
     */
    private function getParamsToFetch()
    {
        $params = [
            'brand_id' => $this->getBrand(),
            'type' => $this->getType(),
            'campaign_id' => $this->getCid(),
            'offer_id' => $this->getOid(),
            'source' => $this->getSource(),
            'default_source' => $this->getDefaultSource(),
            'app_campaign_id' => $this->getAppCid(),
            'app_offer_id' => $this->getAppOid(),
            'app_source' => $this->getAppSource(),
        ];
        return ((http_build_query($params)));
    }

    /**
     * @return mixed
     * @throws Exception
     */
    private function fetch()
    {
        try {
            $curl_handle = curl_init();
            curl_setopt($curl_handle, CURLOPT_URL, static::API_URL . $this->getParamsToFetch());
            curl_setopt($curl_handle, CURLOPT_CONNECTTIMEOUT, 15);
            curl_setopt($curl_handle, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($curl_handle, CURLOPT_USERAGENT, 'FCA LeadHelper - MasterKey Finder');
            $query = curl_exec($curl_handle);
            $httpcode = curl_getinfo($curl_handle, CURLINFO_HTTP_CODE);
            curl_close($curl_handle);

            if ($query === false or $httpcode >= 400) {
                throw new Exception('Unable to get masterkey from FCA API for data: ' . $this->getParamsToFetch());
            }

            return $query;
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), null, $e);
        }
    }

    /**
     * @param $string
     * @return bool
     */
    private function isJson($string)
    {
        json_decode($string);
        return (json_last_error() == JSON_ERROR_NONE);
    }

    /**
     * @param LoggerInterface $logger
     */
    public function setLogger(LoggerInterface $logger)
    {
        $this->logger = $logger;
    }
}
