<?php

namespace FCAPoland\LeadsAPIHelper;

/**
 * Class Request
 *
 * This is common part of all API request (create, update, fetch).
 *
 * @package FCAPoland\LeadsAPIHelper
 */
abstract class Request
{
    const DEFAULT_URL = 'https://api.fcapoland.pl/leads';
    const METHOD_GET = 'GET';
    const METHOD_POST = 'POST';
    const METHOD_PUT = 'PUT';
    const METHOD_DELETE = 'DELETE';

    /**
     * @var resource
     */
    protected $resource;

    /**
     * @var string
     */
    protected $url;

    /**
     * @var string
     */
    private $api_username;

    /**
     * @var string
     */
    private $api_password;

    /**
     * @var array
     */
    protected $query_params = [];

    /**
     * @var string
     */
    protected $body;

    /**
     * @var string HTTP method: GET, POST, PUT, DELETE. Defaults to GET. Child classes may use their own methods.
     */
    protected $method = self::METHOD_GET;

    /**
     * @var array
     */
    private $headers = [
        'Cache-Control' => 'no-cache',
        'Pragma' => 'no-cache',
    ];

    /**
     * @var array
     */
    private $response = [
        'body' => null,
        'code' => null,
    ];

    public function __construct($url = null, $api_username = null, $api_password = null)
    {
        $this->resource = curl_init();
        curl_setopt($this->resource, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($this->resource, CURLOPT_HEADER, false);
        if ($url === null) {
            $url = self::DEFAULT_URL;
        }
        $this->setURL($url);
        $this->setUsername($api_username);
        $this->setPassword($api_password);
        $this->setMethod();
        $this->setContentType();
    }

    /**
     * Actually communicates with the remote API
     *
     * @throws Exception
     * @throws APIException
     */
    public function execute()
    {
        // Set the URL to be requested
        curl_setopt($this->resource, CURLOPT_URL, $this->createURL());
        // Set the Basic HTTP Authorization if username and password are available
        if ($this->api_username and $this->api_password) {
            curl_setopt($this->resource, CURLOPT_USERPWD, $this->api_username . ':' . $this->api_password);
        }
        // Add body to the request, if available
        if ($body = $this->createBody()) {
            curl_setopt($this->resource, CURLOPT_POSTFIELDS, $body);
            $this->headers['Content-Length'] = strlen($body);
        }
        // Set request headers
        curl_setopt($this->resource, CURLOPT_HTTPHEADER, $this->createHeader());
        $this->response['body'] = curl_exec($this->resource);
        $this->response['code'] = curl_getinfo($this->resource, CURLINFO_HTTP_CODE);
        $error = curl_error($this->resource); // Returns the cURL error message or '' (the empty string) for no error.
        curl_close($this->resource);

        if ($error) {
            // Handle cURL errors
            throw new Exception($error);
        }

        if ($this->response['body'] === false) {
            // Even if there is no specific cURL error but `curl_exec` returned boolean `false`, treat this as an error
            throw new Exception('Failed to execute the request');
        }

        if (!is_int($this->response['code']) or $this->response['code'] >= 400) {
            // Finally handle errors signaled by API, like validation or authorization
            $error = $this->response['body'];
            if ($response_decoded = json_decode($this->getResponseBody(), true) and $response_decoded !== null) {
                if (is_array($response_decoded) and array_key_exists('message', $response_decoded)) {
                    $error = is_scalar($response_decoded['message'])
                        ? $response_decoded['message']
                        : json_encode($response_decoded['message']);
                } else {
                    $error = json_encode($response_decoded);
                }
            }
            throw new APIException($error, $this->getResponseCode());
        }
    }

    /**
     * Constructs the complete URL with query params
     *
     * @return string
     */
    protected function createURL()
    {
        $glue = '?';
        if (strpos($this->url, $glue) !== false) {
            $glue = '&';
        }

        return join($glue, array_filter([$this->url, http_build_query($this->query_params)]));
    }

    /**
     * Constructs headers in plain format (strings where key is separated with value using colon)
     *
     * @return array
     */
    protected function createHeader()
    {
        $headers = [];
        foreach ($this->headers as $header => $value) {
            $headers[] = $header . ': ' . $value;
        }
        return $headers;
    }

    /**
     * Creates request body. Can be overriden by specific request types.
     *
     * @return string
     */
    protected function createBody()
    {
        return (string) $this->body;
    }

    /**
     * Sets query params for the API's URL
     *
     * @param array $params
     */
    public function setQueryParams(array $params)
    {
        $this->query_params = $params;
    }

    /**
     * Sets single query param for the API's URL
     *
     * @param $param_key
     * @param $param_value
     */
    public function setQueryParam($param_key, $param_value)
    {
        $this->query_params[$param_key] = $param_value;
    }

    /**
     * Sets the raw body for the request
     *
     * @param $body
     */
    public function setBody($body)
    {
        $this->body = $body;
    }

    /**
     * Returns HTTP Status Code after request execution
     *
     * @return mixed
     */
    public function getResponseCode()
    {
        return $this->response['code'];
    }

    /**
     * Returns body of the response
     *
     * @return mixed
     */
    public function getResponseBody()
    {
        return $this->response['body'];
    }

    /**
     * Sets actual HTTP method to be used for request.
     */
    protected function setMethod()
    {
        switch ($this->method) {
            case self::METHOD_GET:
                break;
            case self::METHOD_POST:
                curl_setopt($this->resource, CURLOPT_POST, true);
                break;
            case self::METHOD_PUT:
            case self::METHOD_DELETE:
            default:
                curl_setopt($this->resource, CURLOPT_CUSTOMREQUEST, strtoupper($this->method));
        }
    }

    /**
     * Sets the type of content that request will use (defined within header fields). Child classes
     * may use this method to customize the content they send. Defaults to `application/json`.
     */
    protected function setContentType()
    {
        $this->headers['Content-Type'] = 'application/json';
    }

    /**
     * @param string $api_username
     */
    public function setUsername($api_username)
    {
        $this->api_username = $api_username;
    }

    /**
     * @param string $api_password
     */
    public function setPassword($api_password)
    {
        $this->api_password = $api_password;
    }

    /**
     * @param string $url
     */
    public function setURL($url)
    {
        $this->url = $url;
    }
}
