<?php

namespace Tests\Functional;

use FCAPoland\LeadsAPIHelper;

/**
 * Class DealersTest
 *
 * These tests require fully functional HTTP server serving API data from any source:
 *  - development
 *  - tests
 *  - production
 *
 * Please check `LEADS_API_URL` constant than can be set in PHPUnit's `bootstrap.php` file.
 *
 * @package Tests\Functional
 */
class CreateLeadTest extends \PHPUnit_Framework_TestCase
{
    /**
     * @var string
     */
    private $url;

    /**
     * @var string
     */
    private $username;

    /**
     * @var string
     */
    private $password;

    protected function setUp()
    {
        parent::setUp();

        if (defined('LEADS_API_URL')) {
            $this->url = LEADS_API_URL;
        }

        if (defined('LEADS_API_USERNAME')) {
            $this->username = LEADS_API_USERNAME;
        }

        if (defined('LEADS_API_PASSWORD')) {
            $this->password = LEADS_API_PASSWORD;
        }
    }

    public function testCreate()
    {
        $lead = new LeadsAPIHelper\Lead();

        $lead->setFirstName('Foo');
        $lead->setLastName('Bar');
        $lead->setType(LeadsAPIHelper\Lead::TYPE_GETAQUOTE);
        $lead->setBrandId(\FCAPoland\LeadsAPIHelper\Lead::BRAND_JEEP);
        $lead->setEmail('foo.bar@foobar.com.cx');
        $lead->setPhoneNumber('612345679');
        $lead->setDealerSincom('0720006.000');
        $lead->setModelId('604');
        $lead->setAppointmentDate('2020-10-21');
        $lead->setOrigin('PhpUnit');
        $lead->setUserAgent('Opera/9.80 (X11; Linux i686; Ubuntu/14.10) Presto/2.12.388 Version/12.16');
        $lead->setMasterkeyId(597); // Note: Masterkey must be valid for given brand
        $lead->setCompany('ACME');
        $lead->setAddress('123 Foobar Str');
        $lead->setZipCode('12-345');
        $lead->setCity('Foobar');
        $lead->setRelatedId(987);
        $lead->setDevice(LeadsAPIHelper\Lead::DEVICE_DESKTOP);
        $lead->setIP('212.43.234.142');
        $lead->setComment('Test Comment');
        $lead->setPrivacy(LeadsAPIHelper\Lead::PRIVACY_TYPE_EMAIL, LeadsAPIHelper\Lead::PRIVACY_ACCEPTED);
        $lead->setPrivacy(LeadsAPIHelper\Lead::PRIVACY_TYPE_PHONE, LeadsAPIHelper\Lead::PRIVACY_UNKNOWN);
        $lead->setPrivacy(LeadsAPIHelper\Lead::PRIVACY_TYPE_SMS, LeadsAPIHelper\Lead::PRIVACY_REFUSED);
        $lead->setDisclaimerId(783);
        $lead->setPrivacy(LeadsAPIHelper\Lead::PRIVACY_TYPE_PROFILING, LeadsAPIHelper\Lead::PRIVACY_REFUSED);

        // Add 'skip-callcenter-verification' query parameter, mark lead to be skipped by call center
        $lead->skipCallCenterVerification();

        // These are invalid for Disclaimer with ID 783:
//        $lead->setPrivacy(LeadsAPIHelper\Lead::PRIVACY_TYPE_MARKETING, LeadsAPIHelper\Lead::PRIVACY_ACCEPTED);
//        $lead->setPrivacy(LeadsAPIHelper\Lead::PRIVACY_TYPE_3RD_PARTY_MARKETING, LeadsAPIHelper\Lead::PRIVACY_ACCEPTED);

        $api_request = new LeadsAPIHelper\Request\Create($this->url, $this->username, $this->password);

        // Attach lead to the request
        $api_request->setLead($lead);

        $api_request->execute();

        $this->assertAttributeEquals(['skip-callcenter-verification' => true], 'query_params', $api_request);
        $this->assertInternalType('int', $lead->getId());
    }

    public function testCreateMissingType()
    {
        $lead = new LeadsAPIHelper\Lead();

        $lead->setFirstName('Foo');
        $lead->setLastName('Bar');
        $lead->setBrandId(\FCAPoland\LeadsAPIHelper\Lead::BRAND_JEEP);
        $lead->setEmail('foo.bar@foobar.com.cx');
        $lead->setPhoneNumber('612345679');
        $lead->setDealerSincom('0720006.000');
        $lead->setModelId('604');
        $lead->setOrigin('PhpUnit');

        $api_request = new LeadsAPIHelper\Request\Create($this->url, $this->username, $this->password);

        // Attach lead to the request
        $api_request->setLead($lead);

        $this->expectException(LeadsAPIHelper\Exception::class);
        $this->expectExceptionMessage('Lead type is required');
        $api_request->execute();

        $this->assertNull($lead->getId());
    }

    public function testCreateInvalidPrivacy()
    {
        $lead = new LeadsAPIHelper\Lead();

        $lead->setFirstName('Foo');
        $lead->setLastName('Bar');
        $lead->setBrandId(\FCAPoland\LeadsAPIHelper\Lead::BRAND_JEEP);
        $lead->setType(LeadsAPIHelper\Lead::TYPE_GETAQUOTE);
        $lead->setEmail('foo.bar@foobar.com.cx');
        $lead->setPhoneNumber('612345679');
        $lead->setDealerSincom('0720006.000');
        $lead->setModelId('604');
        $lead->setOrigin('PhpUnit');
        $lead->setDisclaimerId(723);
        $lead->setPrivacy(LeadsAPIHelper\Lead::PRIVACY_TYPE_MARKETING, LeadsAPIHelper\Lead::PRIVACY_ACCEPTED);
        $lead->setPrivacy(LeadsAPIHelper\Lead::PRIVACY_TYPE_PROFILING, LeadsAPIHelper\Lead::PRIVACY_REFUSED);
        $lead->setPrivacy(LeadsAPIHelper\Lead::PRIVACY_TYPE_3RD_PARTY_MARKETING, LeadsAPIHelper\Lead::PRIVACY_ACCEPTED);

        // If 'SMS' privacy is not defined in the 723 disclaimer, so it should be handled as an exception by the FCA API
        $lead->setPrivacy(LeadsAPIHelper\Lead::PRIVACY_TYPE_SMS, LeadsAPIHelper\Lead::PRIVACY_ACCEPTED);

        $api_request = new LeadsAPIHelper\Request\Create($this->url, $this->username, $this->password);

        // Attach lead to the request
        $api_request->setLead($lead);

        $this->expectException(LeadsAPIHelper\Exception::class);
        $this->expectExceptionMessageRegExp('/Invalid privacy key:.*/');
        $api_request->execute();

        $this->assertInternalType('int', $lead->getId());
    }
}
