<?php

namespace Tests\Unit;

use FCAPoland\LeadsAPIHelper\InvalidArgumentException;
use FCAPoland\LeadsAPIHelper\Lead;
use PHPUnit\Framework\TestCase;

class LeadTest extends TestCase
{
    private $lead_properties_template = [
        'type'          => null,
        'brand_id'      => null,
        'masterkey_id'  => null,
        'first_name'    => null,
        'last_name'     => null,
        'company'       => null,
        'address'       => null,
        'zip_code'      => null,
        'city'          => null,
        'email'         => null,
        'privacy'       => [
            'Email' => null,
            'SMS' => null,
            'Phone' => null,
            'PLThirdParty' => null,
        ],
        'phone_number'  => null,
        'dealer_sincom' => null,
        'model_id'      => null,
        'related_id'    => null,
        'origin'        => null,
        'user_agent'    => null,
        'device'        => null,
        'ip'            => null,
    ];

    public function testPropertiesSetting()
    {
        $lead = new Lead();
        $this->assertAttributeEquals(null, 'first_name', $lead);
        $lead->setFirstName('foo');
        $this->assertAttributeEquals('foo','first_name', $lead);
        $this->assertEquals('foo', $lead->getFirstName());
        $this->assertEquals(
            array_replace($this->lead_properties_template, ['first_name' => 'foo']),
            $lead->getProperties()
        );
    }

    public function testInvalidPropertyTypeSetting()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setFirstName(['foo', 'bar']);
    }

    public function testIdSetGet()
    {
        $lead = new Lead();
        $lead->setId(123);
        $this->assertAttributeEquals(123, 'id', $lead);
        $this->assertEquals(123, $lead->getId());
    }

    public function testSetPrivacySet()
    {
        $privacy_set = [
            Lead::PRIVACY_TYPE_EMAIL => Lead::PRIVACY_ACCEPTED,
            Lead::PRIVACY_TYPE_SMS => Lead::PRIVACY_REFUSED,
            Lead::PRIVACY_TYPE_PHONE => Lead::PRIVACY_UNKNOWN,
        ];
        $lead = new Lead();
        $lead->setPrivacySet($privacy_set);
        $this->assertAttributeEquals(array_merge($privacy_set, [Lead::PRIVACY_TYPE_PLTHIRDPARTY => null]), 'privacy', $lead);
        $this->assertEquals(array_merge($privacy_set, [Lead::PRIVACY_TYPE_PLTHIRDPARTY => null]), $lead->getPrivacySet());
    }

    public function testSetInvalidPrivacySet1()
    {
        $privacy_set = [
            Lead::PRIVACY_TYPE_EMAIL => Lead::PRIVACY_ACCEPTED,
            Lead::PRIVACY_TYPE_SMS => Lead::PRIVACY_REFUSED,
            Lead::PRIVACY_TYPE_PHONE => Lead::PRIVACY_UNKNOWN,
            'foo' => Lead::PRIVACY_ACCEPTED,
        ];
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setPrivacySet($privacy_set);
    }

    public function testSetInvalidPrivacySet2()
    {
        $privacy_set = [
            Lead::PRIVACY_TYPE_EMAIL => Lead::PRIVACY_ACCEPTED,
            Lead::PRIVACY_TYPE_SMS => Lead::PRIVACY_REFUSED,
            Lead::PRIVACY_TYPE_PHONE => Lead::PRIVACY_UNKNOWN,
            Lead::PRIVACY_TYPE_PLTHIRDPARTY => 'foo',
        ];
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setPrivacySet($privacy_set);
    }

    public function testSetPrivacy()
    {
        $lead = new Lead();
        $lead->setPrivacy(Lead::PRIVACY_TYPE_PHONE, Lead::PRIVACY_ACCEPTED);
        $this->assertEquals(
            array_replace(
                $this->lead_properties_template,
                ['privacy' => array_replace($this->lead_properties_template['privacy'], ['Phone' => 1])]
            ),
            $lead->getProperties()
        );
        $this->assertEquals(1, $lead->getPrivacy(Lead::PRIVACY_TYPE_PHONE));
    }

    public function testSetInvalidPrivacy1()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setPrivacy(Lead::PRIVACY_TYPE_PHONE, 'foo');
    }

    public function testSetInvalidPrivacy2()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setPrivacy('foo', Lead::PRIVACY_ACCEPTED);
    }

    public function testSetZipCode()
    {
        $lead = new Lead();
        $lead->setZipCode('12345');
        $this->assertEquals('12-345', $lead->getZipCode());
        $lead->setZipCode('98-765');
        $this->assertEquals('98-765', $lead->getZipCode());
        $lead->setZipCode('foo 45-678 bar');
        $this->assertEquals('45-678', $lead->getZipCode());
    }

    public function testSetInvalidZipCode1()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setZipCode(12345);
    }
    public function testSetInvalidZipCode2()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setZipCode('12--34');
    }

    public function testSetInvalidZipCode3()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setZipCode('123456');
    }

    public function testSetEmail()
    {
        $lead = new Lead();
        $lead->setEmail('foo@bar.com');
        $this->assertEquals('foo@bar.com', $lead->getEmail());
    }

    public function testSetInvalidEmail1()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setEmail('foo@bar');
    }

    public function testSetInvalidEmail2()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setEmail('john doe');
    }

    public function testSetPhoneNumber()
    {
        $lead = new Lead();
        $lead->setPhoneNumber('512345678');
        $this->assertEquals('512345678', $lead->getPhoneNumber());
        $lead->setPhoneNumber('(12)3456789');
        $this->assertEquals('123456789', $lead->getPhoneNumber());
    }

    public function testSetInvalidPhoneNumber1()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setPhoneNumber(512345678);
    }

    public function testSetInvalidPhoneNumber2()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setPhoneNumber('51234567');
    }

    public function testSetDealerSincom()
    {
        $lead = new Lead();
        $lead->setDealerSincom('1234567.123');
        $this->assertEquals('1234567.123', $lead->getDealerSincom());
    }

    public function testSetInvalidDealerSincom1()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setDealerSincom('1234567');
    }

    public function testSetInvalidDealerSincom2()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setDealerSincom('1234567-123');
    }

    public function testSetInvalidDealerSincom3()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setDealerSincom('1234567123');
    }

    public function testSetInvalidDealerSincom4()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setDealerSincom('1234567.12');
    }

    public function testSetInvalidDealerSincom5()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setDealerSincom('123456.123');
    }

    public function testSetInvalidDealerSincom6()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setDealerSincom('1234567.1234');
    }

    public function testSetInvalidDealerSincom7()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setDealerSincom('12345678.123');
    }

    public function testSetIp()
    {
        $lead = new Lead();
        $lead->setIp('91.227.200.210');
        $this->assertEquals('91.227.200.210', $lead->getIp());
    }

    public function testSetInvalidIp1()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setIp('123.456.789.012');
    }

    public function testSetInvalidIp2()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setIp('91227200210');
    }

    public function testSetMasterkeyId()
    {
        $lead = new Lead();
        $lead->setMasterkeyId(123);
        $this->assertEquals(123, $lead->getMasterkeyId());
    }

    public function testSetInvalidMasterkeyId()
    {
        $lead = new Lead();
        $this->expectException(InvalidArgumentException::class);
        $lead->setMasterkeyId('123');
    }

    public function testSetTooLongFirstName()
    {
        $lead = new Lead();
        $too_long_first_name = 'Alias deserunt aut suscipit quae. Et eos placeat aspernatur quia ab. Adipisci' .
            ' explicabo nulla voluptas saepe. Error numquam repellat possimus quia';
        $lead->setFirstName($too_long_first_name);
        $this->assertEquals(mb_substr($too_long_first_name, 0, 50), $lead->getFirstName());
        $this->assertEquals(50, mb_strlen($lead->getFirstName()));
    }
}
