<?php

namespace FCAPoland\LeadsAPIHelper;

use Psr\Log\LoggerInterface;

class Lead
{
    const TYPE_TESTDRIVE = 'TD';
    const TYPE_GETAQUOTE = 'RP';
    const TYPE_REQUEST_INFO = 'RI';
    const TYPE_KEEP_ME_INFORMED = 'KMI';
    const TYPE_NEWSLETTER = 'NL';
    const TYPE_BROCHURE = 'BR';
    const TYPE_VOUCHER = 'VC';
    const TYPE_FINANCIAL = 'FI';
    const TYPE_FIND_MY_VEHICLE = 'FMV';
    const TYPE_PAPER_BROCHURE = 'PBR_PERM';
    const TYPE_FAVORITE = 'FV';
    const TYPE_GET_A_QUOTE_FLEET = 'RF';

    // These privacies was in use before RODO/GDPR came in (25.05.2018)
    const PRIVACY_TYPE_EMAIL = 'Email';
    const PRIVACY_TYPE_SMS = 'SMS';
    const PRIVACY_TYPE_PHONE = 'Phone';
    const PRIVACY_TYPE_PLTHIRDPARTY = 'PLThirdParty';
    const PRIVACY_TYPE_3RD_PARTY_EMAIL = '3rdPartyEmail';
    const PRIVACY_TYPE_3RD_PARTY_SMS = '3rdPartySMS';
    const PRIVACY_TYPE_3RD_PARTY_PHONE = '3rdPartyPhone';

    // These are used as a RODO/GDPR-compliant privacy agreements in favor of `PRIVACY_TYPE_EMAIL`,
    // `PRIVACY_TYPE_SMS`, `PRIVACY_TYPE_PHONE` and `PRIVACY_TYPE_PLTHIRDPARTY`)
    const PRIVACY_TYPE_MARKETING = 'Marketing';
    const PRIVACY_TYPE_PROFILING = 'Profiling';
    const PRIVACY_TYPE_3RD_PARTY_MARKETING = '3rdPartyMarketing';

    const PRIVACY_ACCEPTED = 1;
    const PRIVACY_REFUSED = 0;
    const PRIVACY_UNKNOWN = -1;

    const DEVICE_DESKTOP = 'Desktop';
    const DEVICE_MOBILE = 'Mobile';
    const DEVICE_TABLET = 'Tablet';

    const BRAND_FIAT = '00';
    const BRAND_CAMPER = '11';
    const BRAND_CHRYSLER = '55';
    const BRAND_DODGE = '56';
    const BRAND_JEEP = '57';
    const BRAND_ABARTH = '66';
    const BRAND_LANCIA = '70';
    const BRAND_FIAT_PROFESSIONAL = '77';
    const BRAND_ALFA_ROMEO = '83';
    const BRAND_USED_CARS = '99';

    private $type;
    private $brand_id;
    private $masterkey_id;
    private $first_name;
    private $last_name;
    private $company;
    private $address;
    private $zip_code;
    private $city;
    private $email;
    private $disclaimer_id;
    private $privacy = [
        self::PRIVACY_TYPE_EMAIL               => self::PRIVACY_UNKNOWN,
        self::PRIVACY_TYPE_SMS                 => self::PRIVACY_UNKNOWN,
        self::PRIVACY_TYPE_PHONE               => self::PRIVACY_UNKNOWN,
        self::PRIVACY_TYPE_PLTHIRDPARTY        => self::PRIVACY_UNKNOWN,
        self::PRIVACY_TYPE_3RD_PARTY_EMAIL     => self::PRIVACY_UNKNOWN,
        self::PRIVACY_TYPE_3RD_PARTY_SMS       => self::PRIVACY_UNKNOWN,
        self::PRIVACY_TYPE_3RD_PARTY_PHONE     => self::PRIVACY_UNKNOWN,
        // RODO/GDPR-compliant agreements:
        self::PRIVACY_TYPE_MARKETING           => self::PRIVACY_UNKNOWN,
        self::PRIVACY_TYPE_PROFILING           => self::PRIVACY_UNKNOWN,
        self::PRIVACY_TYPE_3RD_PARTY_MARKETING => self::PRIVACY_UNKNOWN,
    ];
    private $phone_number;
    private $dealer_sincom;
    private $model_id;
    private $appointment_date; // Like testdrive date
    private $related_id;
    private $origin;
    private $user_agent;
    private $device;
    private $ip;

    /**
     * @var int
     */
    private $id;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @param array $properties
     *
     * @throws InvalidArgumentException
     */
    public function setProperties(array $properties)
    {
        foreach ($properties as $property_name => $property_value) {
            switch ($property_name) {
                case 'type':
                    $this->setType($property_value);
                    break;
                case 'brand_id':
                    $this->setBrandId($property_value);
                    break;
                case 'masterkey_id':
                    $this->setMasterkeyId($property_value);
                    break;
                case 'first_name':
                    $this->setFirstName($property_value);
                    break;
                case 'last_name':
                    $this->setLastName($property_value);
                    break;
                case 'company':
                    $this->setCompany($property_value);
                    break;
                case 'address':
                    $this->setAddress($property_value);
                    break;
                case 'zip_code':
                    $this->setZipCode($property_value);
                    break;
                case 'city':
                    $this->setCity($property_value);
                    break;
                case 'email':
                    $this->setEmail($property_value);
                    break;
                case 'disclaimer_id':
                    $this->setDisclaimerId($property_value);
                    break;
                case 'privacy':
                    $this->setPrivacySet($property_value);
                    break;
                case 'phone_number':
                    $this->setPhoneNumber($property_value);
                    break;
                case 'dealer_sincom':
                    $this->setDealerSincom($property_value);
                    break;
                case 'model_id':
                    $this->setModelId($property_value);
                    break;
                case 'appointment_date':
                    $this->setAppointmentDate($property_value);
                    break;
                case 'related_id':
                    $this->setRelatedId($property_value);
                    break;
                case 'origin':
                    $this->setOrigin($property_value);
                    break;
                case 'user_agent':
                    $this->setUserAgent($property_value);
                    break;
                case 'device':
                    $this->setDevice($property_value);
                    break;
                case 'ip':
                    $this->setIP($property_value);
                    break;
                default:
                    throw new InvalidArgumentException('Invalid property ' . $property_name);
            }
        }
    }

    /**
     * @return array
     */
    public function getProperties()
    {
        return [
            'type'             => $this->getType(),
            'brand_id'         => $this->getBrandId(),
            'masterkey_id'     => $this->getMasterkeyId(),
            'first_name'       => $this->getFirstName(),
            'last_name'        => $this->getLastName(),
            'company'          => $this->getCompany(),
            'address'          => $this->getAddress(),
            'zip_code'         => $this->getZipCode(),
            'city'             => $this->getCity(),
            'email'            => $this->getEmail(),
            'disclaimer_id'    => $this->getDisclaimerId(),
            'privacy'          => $this->getPrivacySet(),
            'phone_number'     => $this->getPhoneNumber(),
            'dealer_sincom'    => $this->getDealerSincom(),
            'model_id'         => $this->getModelId(),
            'appointment_date' => $this->getAppointmentDate(),
            'related_id'       => $this->getRelatedId(),
            'origin'           => $this->getOrigin(),
            'user_agent'       => $this->getUserAgent(),
            'device'           => $this->getDevice(),
            'ip'               => $this->getIp(),
        ];
    }

    /**
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @param int $id
     *
     * @throws InvalidArgumentException
     */
    public function setId($id)
    {
        if (!is_int($id)) {
            throw new InvalidArgumentException('Lead ID must be an integer, ' . gettype($id) . ' given');
        }
        $this->id = $id;
    }

    /**
     * @return array
     */
    public function getPrivacySet()
    {
        return $this->privacy;
    }

    /**
     * @param $key
     *
     * @return int|null
     * @throws InvalidArgumentException
     */
    public function getPrivacy($key)
    {
        if (!array_key_exists($key, $this->privacy)) {
            throw new InvalidArgumentException('Invalid privacy: ' . $key);
        }

        return array_key_exists($key, $this->privacy) ? $this->privacy[$key] : null;
    }

    /**
     * @param array $privacy_set
     *
     * @throws InvalidArgumentException
     */
    public function setPrivacySet(array $privacy_set)
    {
        foreach ($privacy_set as $privacy_key => $privacy_value) {
            $this->setPrivacy($privacy_key, $privacy_value);
        }
    }

    /**
     * @param string $key
     * @param int    $value
     *
     * @throws InvalidArgumentException
     */
    public function setPrivacy($key, $value)
    {
        if (!is_string($key)) {
            throw new InvalidArgumentException('Privacy must be specified using string, ' . gettype($key) . ' given');
        }

        if (!array_key_exists($key, $this->privacy)) {
            throw new InvalidArgumentException('Unknown privacy: ' . $key);
        }

        if (!in_array($value, [self::PRIVACY_ACCEPTED, self::PRIVACY_REFUSED, self::PRIVACY_UNKNOWN], true)) {
            throw new InvalidArgumentException('Invalid privacy value: ' . (is_scalar($value) ? $value : json_encode($value)));
        }

        $this->privacy[$key] = $value;
    }

    /**
     * @return int
     */
    public function getDisclaimerId()
    {
        return $this->disclaimer_id;
    }

    public function setDisclaimerId($disclaimer_id)
    {
        if (!is_int($disclaimer_id)) {
            throw new InvalidArgumentException('Disclaimer ID must be an integer, ' . gettype($disclaimer_id) . ' given');
        }
        $this->disclaimer_id = $disclaimer_id;
    }

    /**
     * @return string
     */
    public function getType()
    {
        return $this->type;
    }

    /**
     * @param $type
     *
     * @throws InvalidArgumentException
     */
    public function setType($type)
    {
        if (!is_string($type)) {
            throw new InvalidArgumentException('Lead\'s type must be a string, ' . gettype($type) . ' given');
        }
        $this->type = $type;
    }

    /**
     * @return string
     */
    public function getBrandId()
    {
        return $this->brand_id;
    }

    /**
     * @param string $brand_id
     *
     * @throws InvalidArgumentException
     */
    public function setBrandId($brand_id)
    {
        if (!is_string($brand_id)) {
            throw new InvalidArgumentException('Brand ID must be a string, ' . gettype($brand_id) . ' given');
        }
        $this->brand_id = $brand_id;
    }

    /**
     * @return int
     */
    public function getMasterkeyId()
    {
        return $this->masterkey_id;
    }

    /**
     * @param int $masterkey_id
     *
     * @throws InvalidArgumentException
     */
    public function setMasterkeyId($masterkey_id)
    {
        if (!is_int($masterkey_id)) {
            throw new InvalidArgumentException('Masterkey ID must be an integer, ' . gettype($masterkey_id) . ' given');
        }
        $this->masterkey_id = $masterkey_id;
    }

    /**
     * @param Masterkey $masterkey
     */
    public function setMasterkey(Masterkey $masterkey)
    {
        $this->masterkey_id = $masterkey->getId();
    }

    /**
     * @return string
     */
    public function getFirstName()
    {
        return $this->first_name;
    }

    /**
     * @param string $first_name Value will be truncated to 50 characters
     *
     * @throws InvalidArgumentException
     */
    public function setFirstName($first_name)
    {
        if (!is_string($first_name)) {
            throw new InvalidArgumentException('First name must be a string, ' . gettype($first_name) . ' given');
        }
        $this->first_name = $this->truncate($first_name, 50, 'first name');
    }

    /**
     * @return string
     */
    public function getLastName()
    {
        return $this->last_name;
    }

    /**
     * @param string $last_name Value will be truncated to 50 characters
     *
     * @throws InvalidArgumentException
     */
    public function setLastName($last_name)
    {
        if (!is_string($last_name)) {
            throw new InvalidArgumentException('Last name must be a string, ' . gettype($last_name) . ' given');
        }
        $this->last_name = $this->truncate($last_name, 50, 'last name');
    }

    /**
     * @return string
     */
    public function getCompany()
    {
        return $this->company;
    }

    /**
     * @param string $company Value will be truncated to 200 characters
     *
     * @throws InvalidArgumentException
     */
    public function setCompany($company)
    {
        if (!is_string($company)) {
            throw new InvalidArgumentException('Company name must be a string, ' . gettype($company) . ' given');
        }
        $this->company = $this->truncate($company, 200, 'company name');
    }

    /**
     * @return string
     */
    public function getAddress()
    {
        return $this->address;
    }

    /**
     * @param string $address Value will be truncated to 200 characters
     *
     * @throws InvalidArgumentException
     */
    public function setAddress($address)
    {
        if (!is_string($address)) {
            throw new InvalidArgumentException('Address (street name) must be a string, ' . gettype($address) . ' given');
        }
        $this->address = $this->truncate($address, 200, 'address (street name)');
    }

    /**
     * @return string
     */
    public function getZipCode()
    {
        return $this->zip_code;
    }

    /**
     * @param string $zip_code
     *
     * @throws InvalidArgumentException
     */
    public function setZipCode($zip_code)
    {
        if (!is_string($zip_code)) {
            throw new InvalidArgumentException('ZIP code must be a string, ' . gettype($zip_code) . ' given');
        }
        $sanitized = preg_replace('/\D+/', '', $zip_code);
        if (strlen($sanitized) != 5) {
            throw new InvalidArgumentException('ZIP (postal) code must contain exactly 5 digits (ex. 12-345)');
        }
        $this->zip_code = substr($sanitized, 0, 2) . '-' . substr($sanitized, 2, 3);
    }

    /**
     * @return string
     */
    public function getCity()
    {
        return $this->city;
    }

    /**
     * @param string $city Value will be truncated to 100 characters
     *
     * @throws InvalidArgumentException
     */
    public function setCity($city)
    {
        if (!is_string($city)) {
            throw new InvalidArgumentException('City name must be a string, ' . gettype($city) . ' given');
        }
        $this->city = $this->truncate($city, 100, 'city name');
    }

    /**
     * @return string
     */
    public function getEmail()
    {
        return $this->email;
    }

    /**
     * @param string $email Value will be truncated to 100 characters
     *
     * @throws InvalidArgumentException
     */
    public function setEmail($email)
    {
        if (!is_string($email)) {
            throw new InvalidArgumentException('E-mail address must be a string, ' . gettype($email) . ' given');
        }
        $sanitized_email = filter_var($email, FILTER_SANITIZE_EMAIL);
        if (!filter_var($sanitized_email, FILTER_VALIDATE_EMAIL)) {
            throw new InvalidArgumentException('Invalid e-mail address');
        }
        $this->email = $this->truncate($email, 100, 'e-mail');
    }

    /**
     * @return string
     */
    public function getPhoneNumber()
    {
        return $this->phone_number;
    }

    /**
     * @param string $phone_number Value will be truncated to 20 digits
     *
     * @throws InvalidArgumentException
     */
    public function setPhoneNumber($phone_number)
    {
        if (!is_string($phone_number)) {
            throw new InvalidArgumentException('Phone number must be a string, ' . gettype($phone_number) . ' given');
        }
        $sanitized = preg_replace('/\D+/', '', $phone_number);
        if (strlen($sanitized) < 9) {
            // assuming 9 as minimum phone length
            throw new InvalidArgumentException('Phone number is too short, ' . strlen($sanitized) . ', 9 digits is the minimum');
        }
        $this->phone_number = $this->truncate($sanitized, 20, 'phone number');
    }

    /**
     * @return string
     */
    public function getDealerSincom()
    {
        return $this->dealer_sincom;
    }

    /**
     * @param string $dealer_sincom
     *
     * @throws InvalidArgumentException
     */
    public function setDealerSincom($dealer_sincom)
    {
        if (!is_string($dealer_sincom)) {
            throw new InvalidArgumentException('Dealer SINCOM must be a string, ' . gettype($dealer_sincom) . ' given');
        }
        if (!preg_match('/^\d{7}\.\d{3}$/', $dealer_sincom)) {
            throw new InvalidArgumentException(
                'Invalid dealer SINCOM: ' . $dealer_sincom . ', use 1234567.123 notation'
            );
        }
        $this->dealer_sincom = $dealer_sincom;
    }

    /**
     * @return string
     */
    public function getModelId()
    {
        return $this->model_id;
    }

    /**
     * @param string $model_id
     *
     * @throws InvalidArgumentException
     */
    public function setModelId($model_id)
    {
        if (!is_string($model_id)) {
            throw new InvalidArgumentException('Model ID must be a string, ' . gettype($model_id) . ' given');
        }
        $this->model_id = $model_id;
    }

    /**
     * @return string
     */
    public function getAppointmentDate()
    {
        return $this->appointment_date;
    }

    /**
     * @param string $appointment_date Date in standard format like YYYY-MM-DD
     *
     * @throws InvalidArgumentException
     */
    public function setAppointmentDate($appointment_date)
    {
        if (!is_string($appointment_date)) {
            throw new InvalidArgumentException(
                'Appointment date must be a string representing date convertable to integer timestamp, '
                . gettype($appointment_date) . ' given'
            );
        }
        // Rare case when appointment date is only a year, like '2010'. PHP does not convert this into '2010-01-01'
        // but falls back to the value of `strtotime`'s second parameter `$now`, which by default holds `time()`.
        if (ctype_digit($appointment_date) and strlen($appointment_date) == 4) {
            $appointment_date = $appointment_date . '-01-01';
        }
        $appointment_date_int = strtotime($appointment_date);
        if ($appointment_date_int === false) {
            throw new InvalidArgumentException(
                'Could not convert appointment date: ' . $appointment_date . ' to integer timestamp.'
            );
        }
        $appointment_date = $appointment_date_int;
        unset($appointment_date_int);
        // Convert the timestamp back to the only accepted format: YYYY-MM-DD.
        $this->appointment_date = date('Y-m-d', $appointment_date);
    }

    /**
     * @return int
     */
    public function getRelatedId()
    {
        return $this->related_id;
    }

    /**
     * @param int $related_id
     *
     * @throws InvalidArgumentException
     */
    public function setRelatedId($related_id)
    {
        if (!is_int($related_id)) {
            throw new InvalidArgumentException('Related ID must be an integer, ' . gettype($related_id) . ' given');
        }
        $this->related_id = $related_id;
    }

    /**
     * @return string
     */
    public function getOrigin()
    {
        return $this->origin;
    }

    /**
     * @param string $origin Value will be truncated to 500 characters
     *
     * @throws InvalidArgumentException
     */
    public function setOrigin($origin)
    {
        if (!is_string($origin)) {
            throw new InvalidArgumentException('Lead\'s origin must be a string, ' . gettype($origin) . ' given');
        }
        $this->origin = $this->truncate($origin, 500, 'origin');
    }

    /**
     * @return string
     */
    public function getUserAgent()
    {
        return $this->user_agent;
    }

    /**
     * @param string $user_agent Value will be truncated to 255 characters
     *
     * @throws InvalidArgumentException
     */
    public function setUserAgent($user_agent)
    {
        if (!is_string($user_agent)) {
            throw new InvalidArgumentException('User agent definition must be a string, ' . gettype($user_agent) . ' given');
        }
        $this->user_agent = $this->truncate($user_agent, 255, 'user agent');
    }

    /**
     * @return string
     */
    public function getDevice()
    {
        return $this->device;
    }

    /**
     * @param string $device Value will be truncated to 8 characters. Choose one `Lead` class constants prefixed with
     *                       `DEVICE_`
     *
     * @throws InvalidArgumentException
     */
    public function setDevice($device)
    {
        if (!is_string($device)) {
            throw new InvalidArgumentException('Device name must be a string, ' . gettype($device) . ' given');
        }
        $this->device = $this->truncate($device, 8, 'device');
    }

    /**
     * @return string
     */
    public function getIp()
    {
        return $this->ip;
    }

    /**
     * @param string $ip
     *
     * @throws InvalidArgumentException
     */
    public function setIp($ip)
    {
        if (!is_string($ip)) {
            throw new InvalidArgumentException('IP address must be given as string, ' . gettype($ip) . ' given');
        }
        if (!filter_var($ip, FILTER_VALIDATE_IP)) {
            throw new InvalidArgumentException('Invalid IP address');
        }
        $this->ip = $ip;
    }

    /**
     * Attaches PSR-3 compatibile logger to the module allowing it to send messages to application logging system
     *
     * @param LoggerInterface $logger
     */
    public function setLogger(LoggerInterface $logger)
    {
        $this->logger = $logger;
    }

    /**
     * Truncates given value to it's max length producing warning message if logger is available
     *
     * @param $value
     * @param $max_length
     * @param $field_name
     * @return string
     */
    private function truncate($value, $max_length, $field_name)
    {
        if (mb_strlen($value) > $max_length) {
            if ($this->logger instanceof LoggerInterface) {
                $this->logger->warning(
                    'The value of "' . $field_name . '" should not exceed ' . $max_length .
                    ' characters. Redundant characters will be silently truncated'
                );
            }
            $value = mb_substr($value, 0, $max_length);
        }

        return $value;
    }
}
